"use strict";
const lib_1 = require("../lib");
const util_1 = require("./util");
class MyResource extends lib_1.Resource {
    constructor(scope, id, physicalName) {
        super(scope, id, { physicalName });
        this.arn = this.getResourceArnAttribute('simple-arn', {
            region: '',
            account: '',
            resource: 'my-resource',
            resourceName: this.physicalName,
            service: 'myservice',
        });
        this.name = this.getResourceNameAttribute('simple-name');
        new lib_1.CfnResource(this, 'Resource', {
            type: 'My::Resource',
            properties: {
                resourceName: this.physicalName,
            },
        });
    }
}
module.exports = {
    'CrossEnvironmentToken': {
        'can reference an ARN with a fixed physical name directly in a different account'(test) {
            // GIVEN
            const app = new lib_1.App();
            const stack1 = new lib_1.Stack(app, 'Stack1', {
                env: {
                    account: '123456789012',
                    region: 'bermuda-triangle-1337',
                },
            });
            const myResource = new MyResource(stack1, 'MyResource', 'PhysicalName');
            const stack2 = new lib_1.Stack(app, 'Stack2', {
                env: {
                    account: '234567890123',
                    region: 'bermuda-triangle-42',
                },
            });
            // WHEN
            new lib_1.CfnOutput(stack2, 'Output', {
                value: myResource.arn,
            });
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack2), {
                Outputs: {
                    Output: {
                        Value: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':myservice:::my-resource/PhysicalName',
                                ],
                            ],
                        },
                    },
                },
            });
            test.done();
        },
        'can reference a fixed physical name directly in a different account'(test) {
            // GIVEN
            const app = new lib_1.App();
            const stack1 = new lib_1.Stack(app, 'Stack1', {
                env: {
                    account: '123456789012',
                    region: 'bermuda-triangle-1337',
                },
            });
            const stack2 = new lib_1.Stack(app, 'Stack2', {
                env: {
                    account: '234567890123',
                    region: 'bermuda-triangle-42',
                },
            });
            // WHEN
            const myResource = new MyResource(stack1, 'MyResource', 'PhysicalName');
            new lib_1.CfnOutput(stack2, 'Output', {
                value: myResource.name,
            });
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack2), {
                Outputs: {
                    Output: {
                        Value: 'PhysicalName',
                    },
                },
            });
            test.done();
        },
        'can reference an ARN with an assigned physical name directly in a different account'(test) {
            // GIVEN
            const app = new lib_1.App();
            const stack1 = new lib_1.Stack(app, 'Stack1', {
                env: {
                    account: '123456789012',
                    region: 'bermuda-triangle-1337',
                },
            });
            const myResource = new MyResource(stack1, 'MyResource', lib_1.PhysicalName.GENERATE_IF_NEEDED);
            const stack2 = new lib_1.Stack(app, 'Stack2', {
                env: {
                    account: '234567890123',
                    region: 'bermuda-triangle-42',
                },
            });
            // WHEN
            new lib_1.CfnOutput(stack2, 'Output', {
                value: myResource.arn,
            });
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack2), {
                Outputs: {
                    Output: {
                        Value: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':myservice:::my-resource/stack1stack1myresourcec54ced43683ebf9a3c4c',
                                ],
                            ],
                        },
                    },
                },
            });
            test.done();
        },
        'can reference an assigned physical name directly in a different account'(test) {
            // GIVEN
            const app = new lib_1.App();
            const stack1 = new lib_1.Stack(app, 'Stack1', {
                env: {
                    account: '123456789012',
                    region: 'bermuda-triangle-1337',
                },
            });
            const stack2 = new lib_1.Stack(app, 'Stack2', {
                env: {
                    account: '234567890123',
                    region: 'bermuda-triangle-42',
                },
            });
            // WHEN
            const myResource = new MyResource(stack1, 'MyResource', lib_1.PhysicalName.GENERATE_IF_NEEDED);
            new lib_1.CfnOutput(stack2, 'Output', {
                value: myResource.name,
            });
            // THEN
            test.deepEqual(util_1.toCloudFormation(stack2), {
                Outputs: {
                    Output: {
                        Value: 'stack1stack1myresourcec54ced43683ebf9a3c4c',
                    },
                },
            });
            test.done();
        },
    },
    'cannot reference a deploy-time physical name across environments'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1', {
            env: {
                account: '123456789012',
                region: 'bermuda-triangle-1337',
            },
        });
        const stack2 = new lib_1.Stack(app, 'Stack2', {
            env: {
                account: '234567890123',
                region: 'bermuda-triangle-42',
            },
        });
        // WHEN
        const myResource = new MyResource(stack1, 'MyResource');
        new lib_1.CfnOutput(stack2, 'Output', {
            value: myResource.name,
        });
        // THEN
        test.throws(() => util_1.toCloudFormation(stack2), /Cannot use resource 'Stack1\/MyResource' in a cross-environment fashion/);
        test.done();
    },
    'cross environment when stack is a substack'(test) {
        const app = new lib_1.App();
        const parentStack = new lib_1.Stack(app, 'ParentStack', {
            env: { account: '112233', region: 'us-east-1' },
        });
        const childStack = new lib_1.Stack(parentStack, 'ChildStack', {
            env: { account: '998877', region: 'eu-west-2' },
        });
        const childResource = new MyResource(childStack, 'ChildResource', lib_1.PhysicalName.GENERATE_IF_NEEDED);
        new lib_1.CfnResource(parentStack, 'ParentResource', {
            type: 'Parent::Resource',
            properties: {
                RefToChildResource: childResource.name,
            },
        });
        const assembly = app.synth();
        test.deepEqual(assembly.getStackByName(parentStack.stackName).template, {
            Resources: {
                ParentResource: {
                    Type: 'Parent::Resource',
                    Properties: {
                        RefToChildResource: 'parentstackchildstack83c5ackchildresource852877eeb919bda2008e',
                    },
                },
            },
        });
        test.deepEqual(assembly.getStackByName(childStack.stackName).template, {
            Resources: {
                ChildResource8C37244D: {
                    Type: 'My::Resource',
                    Properties: {
                        resourceName: 'parentstackchildstack83c5ackchildresource852877eeb919bda2008e',
                    },
                },
            },
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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