# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['dbt_osmosis', 'dbt_osmosis.core']

package_data = \
{'': ['*']}

install_requires = \
['GitPython>=3.1.27,<4.0.0',
 'bottle>=0.12.23,<0.13.0',
 'click>7',
 'dbt-client>=0.1.2,<0.2.0',
 'dbt-core>=1',
 'feedparser>=6.0.10,<7.0.0',
 'graphviz>=0.17',
 'orjson>=3.8.0,<4.0.0',
 'pydantic>=1.9.1,<2.0.0',
 'pydot>=1.4.2',
 'rich>=10',
 'ruamel.yaml>=0.17',
 'scipy>=1.8.1,<2.0.0',
 'streamlit-ace>=0.1.0',
 'streamlit-aggrid>=0.2.2',
 'streamlit-agraph>=0.0.35',
 'streamlit-pandas-profiling>=0.1.2',
 'streamlit>=1.0.0']

extras_require = \
{'dbt-bigquery': ['dbt-bigquery>=1'],
 'dbt-duckdb': ['duckcli==0.0.1', 'dbt-duckdb>=1.1.2,<1.2.0'],
 'dbt-sqlite': ['dbt-sqlite>=1.1.2,<2.0.0']}

entry_points = \
{'console_scripts': ['dbt-osmosis = dbt_osmosis.main:cli',
                     'dbt-osmosis-proxy = dbt_osmosis.server:cli']}

setup_kwargs = {
    'name': 'dbt-osmosis',
    'version': '0.7.11',
    'description': 'This package exposes an interactive workbench for dbt modelling and serves to cascadingly populate column level documentation, build & conform schema files, and audit coverage.',
    'long_description': '# dbt-osmosis\n\n<!--![GitHub Actions](https://github.com/z3z1ma/dbt-osmosis/actions/workflows/master.yml/badge.svg)-->\n\n![PyPI](https://img.shields.io/pypi/v/dbt-osmosis)\n![Downloads](https://pepy.tech/badge/dbt-osmosis)\n![License: Apache 2.0](https://img.shields.io/badge/License-Apache_2.0-green.svg)\n![black](https://img.shields.io/badge/code%20style-black-000000.svg)\n[![Streamlit App](https://static.streamlit.io/badges/streamlit_badge_black_white.svg)](https://z3z1ma-dbt-osmosis-srcdbt-osmosisapp-4y67qs.streamlitapp.com/)\n\n\n## Primary Objectives\n\nHello and welcome to the project! [dbt-osmosis](https://github.com/z3z1ma/dbt-osmosis) 🌊 serves to enhance the developer experience significantly. We do this through providing 3 core features:\n\n1. Automated schema YAML management (minimize developers repetitive tasks)\n\n2. Workbench for dbt Jinja SQL (maximize developers dbt SQL authoring efficiency + learning + testing)\n\n3. Diffs for data model outputs to model outputs across git revisions (optimize developers observability during iteration)\n\n\nWhen combined with an IDE such as VS Code, developers can work with renewed efficiency, enjoyment, and effectiveness throughout their days. \n\n\n[Workbench Reference](#Workbench)\n\n[CLI Reference](#CLI)\n\n____\n\n\n## Workbench\n\nDemo the workbench 👇 \n\n[![Streamlit App](https://static.streamlit.io/badges/streamlit_badge_black_white.svg)](https://z3z1ma-dbt-osmosis-srcdbt-osmosisapp-4y67qs.streamlitapp.com/)\n\n \n```sh\n# Command to start server\ndbt-osmosis workbench\n```\n\nPress "r" to reload the workbench at any time.\n\n\n\n✔️ dbt Editor with instant dbt compilation side-by-side or pivoted\n\n✔️ Full control over model and workbench theme, light and dark mode\n\n✔️ Query Tester, test the model you are working on for instant feedback\n\n✔️ Data Profiler (leverages pandas-profiling)\n\n\n**Editor** \n\nThe editor is able to compile models with control+enter or dynamically as you type. Its speedy! You can choose any target defined in your profiles yml for compilation and execution.\n\n![editor](/screenshots/osmosis_editor_main.png?raw=true "dbt-osmosis Workbench")\n\nYou can pivot the editor for a fuller view while workbenching some dbt SQL.\n\n![pivot](/screenshots/osmosis_editor_pivot.png?raw=true "dbt-osmosis Pivot Layout")\n\n\n**Test Query**\n\nTest dbt models as you work against whatever profile you have selected and inspect the results. This allows very fast iterative feedback loops not possible with VS Code alone.\n\n![test-model](/screenshots/osmosis_tester.png?raw=true "dbt-osmosis Test Model")\n\n**Profile Model Results**\n\nProfile your datasets on the fly while you develop without switching context. Allows for more refined interactive data modelling when dataset fits in memory.\n\n![profile-data](/screenshots/osmosis_profile.png?raw=true "dbt-osmosis Profile Data")\n\n\n**Useful Links and RSS Feed**\n\nSome useful links and RSS feeds at the bottom. 🤓\n\n![profile-data](/screenshots/osmosis_links.png?raw=true "dbt-osmosis Profile Data")\n\n____\n\n\n## CLI\n\ndbt-osmosis is ready to use as-is. To get familiar, you should run it on a fresh branch and ensure everything is backed in source control. Enjoy!\n\nYou should set a base config in your dbt_project.yml and ensure any models within the scope of your execution plan will inherit a config/preference. Example below.\n\n```yaml\nmodels:\n\n    your_dbt_project:\n\n        # This config will apply to your whole project\n        +dbt-osmosis: "schema/model.yml"\n\n        staging:\n\n            # This config will apply to your staging directory\n            +dbt-osmosis: "folder.yml"\n\n            +tags: \n                - "staged"\n\n            +materialized: view\n\n            monday:\n                intermediate:\n                    +materialized: ephemeral\n\n        marts:\n\n            # Underscore prefixed model name as recommended in dbt best practices\n            +dbt-osmosis: "_model.yml"\n\n            +tags: \n                - "mart"\n\n            supply_chain: \n```\n\nTo use dbt-osmosis, simply run the following:\n\n```bash\n# Install\npip install dbt-osmosis\n# Alternatively\npipx install dbt-osmosis dbt-<adapter>\n\n\n# This command executes all tasks in preferred order and is usually all you need\n\ndbt-osmosis run --project-dir /path/to/dbt/project --target prod\n\n\n# Inherit documentation in staging/salesforce/ & sync \n# schema yaml columns with database columns\n\ndbt-osmosis document --project-dir /path/to/dbt/project --target prod --fqn staging.salesforce\n\n\n# Reorganize marts/operations/ & inject undocumented models \n# into schema files or create new schema files as needed\n\ndbt-osmosis compose --project-dir /path/to/dbt/project --target prod --fqn marts.operations\n\n\n# Open the dbt-osmosis workbench\n\ndbt-osmosis workbench\n\n\n# Diff a model from git HEAD to revision on disk\n\ndbt-osmosis diff -m int_account_events --pk \'concat(account_id, date_day)\' --output bar\n```\n\n## Features\n\n### Standardize organization of schema files (and provide ability to define and conform with code)\n\n- Config can be set on per directory basis if desired utilizing `dbt_project.yml`, all models which are processed require direct or inherited config `+dbt-osmosis:`. If even one dir is missing the config, we close gracefully and inform user to update dbt_project.yml. No assumed defaults. Placing our config under your dbt project name in `models:` is enough to set a default for the project since the config applies to all subdirectories. \n\n    Note: You can **change these configs as often as you like** or try them all, dbt-osmosis will take care of restructuring your project schema files-- _no human effort required_. \n\n    A directory can be configured to conform to any one of the following standards:\n\n    - Can be one schema file to one model file sharing the same name and directory ie. \n\n            staging/\n                stg_customer.sql\n                stg_customer.yml\n                stg_order.sql\n                stg_order.yml\n\n        - `+dbt-osmosis: "model.yml"`\n\n    - Can be one schema file per directory wherever model files reside named schema.yml, ie.\n\n            staging/\n                schema.yml\n                stg_customer.sql\n                stg_order.sql\n\n        - `+dbt-osmosis: "schema.yml"`\n    - Can be one schema file per directory wherever model files reside named after its containing folder, ie. \n\n            staging/\n                stg_customer.sql\n                stg_order.sql\n                staging.yml\n\n        - `+dbt-osmosis: "folder.yml"`\n\n    - Can be one schema file to one model file sharing the same name _nested_ in a schema subdir wherever model files reside, ie. \n\n            staging/\n                stg_order.sql\n                stg_customer.sql\n                schema/\n                    stg_customer.yml\n                    stg_order.yml\n\n        - `+dbt-osmosis: "schema/model.yml"`\n\n    - Can be one schema file to one model file sharing the same name and directory, models prefixed with underscore for IDE sorting ie. \n\n            staging/\n                _stg_customer.yml\n                _stg_order.yml\n                stg_customer.sql\n                stg_order.sql\n\n        - `+dbt-osmosis: "_model.yml"`\n\n### Build and Inject Non-documented models\n\n- Injected models will automatically conform to above config per directory based on location of model file. \n\n- This means you can focus fully on modelling; and documentation, including yaml updates or creation, will automatically follow at any time with simple invocation of dbt-osmosis\n\n### Propagate existing column level documentation downward to children\n\n- Build column level knowledge graph accumulated and updated from furthest identifiable origin (ancestors) to immediate parents\n\n- Will automatically populate undocumented columns of the same name with passed down knowledge accumulated within the context of the models upstream dependency tree\n\n- This means you can freely generate models and all columns you pull into the models SQL that already have been documented will be automatically learned/propagated. Again the focus for analysts is almost fully on modelling and yaml work is an afterthought / less heavy of a manual lift.\n\n### Order Matters\n\nIn a full run [ `dbt-osmosis run` ] we will:\n\n1. Conform dbt project\n    - Configuration lives in `dbt_project.yml` --> we require our config to run, can be at root level of `models:` to apply a default convention to a project \n    or can be folder by folder, follows dbt config resolution where config is overridden by scope. \n    Config is called `+dbt-osmosis: "folder.yml" | "schema.yml" | "model.yml" | "schema/model.yml" | "_model.yml"`\n2. Bootstrap models to ensure all models exist\n3. Recompile Manifest\n4. Propagate definitions downstream to undocumented models solely within the context of each models dependency tree\n\n\n## Python API\n\nThough each core function is useful enough to stand as its own package, dbt osmosis sits as a unified interface primarily because all of these functions are built off of the same core API structures in the dbt osmosis package. dbt osmosis provides one of the cleanest interfaces to interacting with dbt if you aren\'t keen to play with dbt on-the-rails (like me) or you want to extend what osmosis can do.\n\n```python\n# Programmatic Examples:\nfrom dbt_osmosis.core import DbtOsmosis\nfrom dbt_osmosis.diff import diff_and_print_to_console\n\nrunner = DbtOsmosis(\n    project_dir="/Users/alexanderbutler/Documents/harness/analytics-pipelines/projects/meltano/harness/transform",\n    dry_run=True,\n    target="prod",\n)\n\n\n# Some dbt osmosis YAML management 📜\n\n# review the generated plan\nrunner.pretty_print_restructure_plan(runner.draft_project_structure_update_plan())\n\n# organize your dbt project based on declarative config\nrunner.commit_project_restructure_to_disk()\n\n# propagate column level documentation down the DAG\nrunner.propagate_documentation_downstream()\n\n\n# Console utilities 📺\n\n# leverage git to diff the OUTPUT of a model from git HEAD \n# to your revision on disk to safely audit changes as you work\ndiff_and_print_to_console("fct_sales", pk="order_id", runner=runner)  \n\n\n# Massively simplified dbt interfaces you likely won\'t find elsewhere 👏\n\n# execute macros through a simple interface without subprocesses\nrunner.execute_macro(\n    "create_schema",\n    kwargs={"relation": relation},\n)\n\n# compile SQL as easy as this 🤟\nrunner.compile_sql("select * from {{ ref(\'stg_salesforce__users\') }}")\n\n# run SQL too\nadapter_resp, table = runner.execute_sql(\n    "select * from {{ ref(\'stg_salesforce__users\') }}", \n    compile=True, \n    fetch=True,\n)\n```\n\n## Roadmap\n\nThese features are being actively developed and will be merged into the next few minor releases\n\n1. Extend git diff functionality to pin revisions in the warehouse  \n2. Complete build out of `sources` tools.\n3. Add `--min-cov` flag to audit task and to workbench\n4. Add interactive documentation flag that engages user to documents ONLY progenitors and novel columns for a subset of models (the most optimized path to full documentation coverage feasible)\n5. Add `impact` command that allows us to leverage our resolved column level progenitors for ad hoc impact analysis\n',
    'author': 'z3z1ma',
    'author_email': 'butler.alex2010@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/z3z1ma/dbt-osmosis',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
