#!/usr/bin/env python

"""mdpo2html command line interface."""

import argparse
import sys

from mdpo.cli import add_common_cli_arguments, parse_list_cli_argument
from mdpo.mdpo2html import markdown_pofile_to_html


DESCRIPTION = (
    'HTML-produced-from-Markdown files translator using pofiles'
    ' as reference.'
)


def build_parser():
    parser = argparse.ArgumentParser(description=DESCRIPTION, add_help=False)
    add_common_cli_arguments(parser)
    parser.add_argument(
        'filepath_or_content', metavar='FILEPATH_OR_CONTENT',
        nargs='*',
        help='HTML filepath or content to translate. If not provided, will be'
             ' read from STDIN.',
    )
    parser.add_argument(
        '-p', '--pofiles', metavar='POFILES',
        help='Glob matching a set of PO files from where to extract references'
             ' to make the replacements translating strings.',
    )
    parser.add_argument(
        '-i', '--ignore', dest='ignore', default=[],
        help='Filepaths to ignore when ``--pofiles`` argument value is a glob,'
             ' as a list of comma separated values.',
        metavar='PATH_1,PATH_2...',
    )
    parser.add_argument(
        '-s', '--save', dest='save', default=None,
        help='Saves the output content in file whose path is specified at this'
             ' parameter.', metavar='PATH',
    )
    return parser


def parse_options(args):
    parser = build_parser()
    if '-h' in args or '--help' in args:
        parser.print_help()
        sys.exit(0)
    opts = parser.parse_args(args)

    if not sys.stdin.isatty():
        opts.filepath_or_content = sys.stdin.read().strip('\n')
    elif isinstance(opts.filepath_or_content, list):
        opts.filepath_or_content = opts.filepath_or_content[0]
    if opts.ignore:
        opts.ignore = parse_list_cli_argument(opts.ignore)

    return opts


def run(args=[]):
    opts = parse_options(args)

    output = markdown_pofile_to_html(
        opts.filepath_or_content, opts.pofiles,
        ignore=opts.ignore, save=opts.save,
    )

    if not opts.quiet and not opts.save:
        sys.stdout.write('%s\n' % output)

    return (output, 0)


def main():
    sys.exit(run(args=sys.argv[1:])[1])  # pragma: no cover


if __name__ == '__main__':
    main()
