"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const codebuild = require("@aws-cdk/aws-codebuild");
const codePipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_DEFAULT_REGION,
            account: process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'my-codepipeline-badge-notification-stack', { env });
        const pipeline = this.createCodePipeline(stack);
        const codePipelineStatus = new index_1.CodePipelineBadgeNotification(stack, 'CodePipelineBadgeNotification', {
            pipelineArn: pipeline.pipelineArn,
            gitHubTokenFromSecretsManager: {
                secretsManagerArn: `arn:aws:secretsmanager:ap-northeast-1:${cdk.Aws.ACCOUNT_ID}:secret:codepipeline/lambda/github-token-YnCnne`,
                secretKey: 'codepipeline/lambda/github-token',
            },
            notification: {
                stageName: 'production',
                // ssmSlackWebHookUrl: 'ssm_slack_webhook',
                ssmGoogleChatWebHookUrl: '/chat/google/webhook',
                ssmTelegramWebHookUrl: '/chat/telegram/webhook',
            },
        });
        new cdk.CfnOutput(stack, 'BadgeUrl', {
            value: codePipelineStatus.badgeUrl,
        });
        new cdk.CfnOutput(stack, 'CodePipelineLink', {
            value: codePipelineStatus.codePipelineLink,
        });
        app.synth();
        this.stack = [stack];
    }
    createCodePipeline(stack) {
        const pipeline = new codePipeline.Pipeline(stack, 'TestPipeline', {
            pipelineName: 'integTestCodePipeline',
            crossAccountKeys: false,
        });
        const sourceOutput = new codePipeline.Artifact();
        pipeline.addStage({
            stageName: 'Source',
            actions: [
                new codepipeline_actions.CodeStarConnectionsSourceAction({
                    actionName: 'GitHub_Source',
                    owner: 'kimisme9386',
                    repo: 'cdk-codepipeline-badge-notification',
                    output: sourceOutput,
                    connectionArn: `arn:aws:codestar-connections:ap-northeast-1:${cdk.Aws.ACCOUNT_ID}:connection/e97c0228-6aee-46df-a0a5-8ddbd3c94679`,
                    variablesNamespace: 'GitHubSourceVariables',
                    branch: 'feature/google-chat-notification',
                    codeBuildCloneOutput: true,
                }),
            ],
        });
        const project = this.createCodeBuildProjectWithinCodePipeline(stack);
        const afterBuildArtifact = new codePipeline.Artifact();
        pipeline.addStage({
            stageName: 'Build',
            actions: [
                new codepipeline_actions.CodeBuildAction({
                    actionName: 'AWS_CodeBuild',
                    input: sourceOutput,
                    project: project,
                    type: codepipeline_actions.CodeBuildActionType.BUILD,
                    outputs: [afterBuildArtifact],
                }),
            ],
        });
        return pipeline;
    }
    createCodeBuildProjectWithinCodePipeline(stack) {
        const project = new codebuild.PipelineProject(stack, 'CodeBuildWithinCodePipeline', {
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    shell: 'bash',
                },
                phases: {
                    build: {
                        'on-failure': 'ABORT',
                        'commands': [
                            'echo "test build"',
                        ],
                    },
                },
                artifacts: {
                    files: 'imagedefinitions.json',
                },
            }),
            environment: {
                buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                computeType: codebuild.ComputeType.SMALL,
                privileged: true,
            },
            cache: codebuild.Cache.local(codebuild.LocalCacheMode.DOCKER_LAYER),
        });
        return project;
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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