"use strict";
// Implementation of metric patterns
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Base class for patterns that only match JSON log events.
 */
class JsonPattern {
    // This is a separate class so we have some type safety where users can't
    // combine text patterns and JSON patterns with an 'and' operation.
    constructor(jsonPatternString) {
        this.jsonPatternString = jsonPatternString;
    }
    get logPatternString() {
        return '{ ' + this.jsonPatternString + ' }';
    }
}
exports.JsonPattern = JsonPattern;
/**
 * A collection of static methods to generate appropriate ILogPatterns
 */
class FilterPattern {
    /**
     * Use the given string as log pattern.
     *
     * See https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     * for information on writing log patterns.
     *
     * @param logPatternString The pattern string to use.
     */
    static literal(logPatternString) {
        return new LiteralLogPattern(logPatternString);
    }
    /**
     * A log pattern that matches all events.
     */
    static allEvents() {
        return new LiteralLogPattern("");
    }
    /**
     * A log pattern that matches if all the strings given appear in the event.
     *
     * @param terms The words to search for. All terms must match.
     */
    static allTerms(...terms) {
        return new TextLogPattern([terms]);
    }
    /**
     * A log pattern that matches if any of the strings given appear in the event.
     *
     * @param terms The words to search for. Any terms must match.
     */
    static anyTerm(...terms) {
        return new TextLogPattern(terms.map(t => [t]));
    }
    /**
     * A log pattern that matches if any of the given term groups matches the event.
     *
     * A term group matches an event if all the terms in it appear in the event string.
     *
     * @param termGroups A list of term groups to search for. Any one of the clauses must match.
     */
    static anyTermGroup(...termGroups) {
        return new TextLogPattern(termGroups);
    }
    /**
     * A JSON log pattern that compares string values.
     *
     * This pattern only matches if the event is a JSON event, and the indicated field inside
     * compares with the string value.
     *
     * Use '$' to indicate the root of the JSON structure. The comparison operator can only
     * compare equality or inequality. The '*' wildcard may appear in the value may at the
     * start or at the end.
     *
     * For more information, see:
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     * @param comparison Comparison to carry out. Either = or !=.
     * @param value The string value to compare to. May use '*' as wildcard at start or end of string.
     */
    static stringValue(jsonField, comparison, value) {
        return new JSONStringPattern(jsonField, comparison, value);
    }
    /**
     * A JSON log pattern that compares numerical values.
     *
     * This pattern only matches if the event is a JSON event, and the indicated field inside
     * compares with the value in the indicated way.
     *
     * Use '$' to indicate the root of the JSON structure. The comparison operator can only
     * compare equality or inequality. The '*' wildcard may appear in the value may at the
     * start or at the end.
     *
     * For more information, see:
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     * @param comparison Comparison to carry out. One of =, !=, <, <=, >, >=.
     * @param value The numerical value to compare to
     */
    static numberValue(jsonField, comparison, value) {
        return new JSONNumberPattern(jsonField, comparison, value);
    }
    /**
     * A JSON log pattern that matches if the field exists and has the special value 'null'.
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     */
    static isNull(jsonField) {
        return new JSONPostfixPattern(jsonField, 'IS NULL');
    }
    /**
     * A JSON log pattern that matches if the field does not exist.
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     */
    static notExists(jsonField) {
        return new JSONPostfixPattern(jsonField, 'NOT EXISTS');
    }
    /**
     * A JSON log patter that matches if the field exists.
     *
     * This is a readable convenience wrapper over 'field = *'
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     */
    static exists(jsonField) {
        return new JSONStringPattern(jsonField, '=', '*');
    }
    /**
     * A JSON log pattern that matches if the field exists and equals the boolean value.
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     * @param value The value to match
     */
    static booleanValue(jsonField, value) {
        return new JSONPostfixPattern(jsonField, value ? 'IS TRUE' : 'IS FALSE');
    }
    /**
     * A JSON log pattern that matches if all given JSON log patterns match
     */
    static all(...patterns) {
        if (patterns.length === 0) {
            throw new Error('Must supply at least one pattern, or use allEvents() to match all events.');
        }
        if (patterns.length === 1) {
            return patterns[0];
        }
        return new JSONAggregatePattern('&&', patterns);
    }
    /**
     * A JSON log pattern that matches if any of the given JSON log patterns match
     */
    static any(...patterns) {
        if (patterns.length === 0) {
            throw new Error('Must supply at least one pattern');
        }
        if (patterns.length === 1) {
            return patterns[0];
        }
        return new JSONAggregatePattern('||', patterns);
    }
    /**
     * A space delimited log pattern matcher.
     *
     * The log event is divided into space-delimited columns (optionally
     * enclosed by "" or [] to capture spaces into column values), and names
     * are given to each column.
     *
     * '...' may be specified once to match any number of columns.
     *
     * Afterwards, conditions may be added to individual columns.
     *
     * @param columns The columns in the space-delimited log stream.
     */
    static spaceDelimited(...columns) {
        return SpaceDelimitedTextPattern.construct(columns);
    }
}
exports.FilterPattern = FilterPattern;
/**
 * Use a string literal as a log pattern
 */
class LiteralLogPattern {
    constructor(logPatternString) {
        this.logPatternString = logPatternString;
    }
}
/**
 * Search for a set of set of terms
 */
class TextLogPattern {
    constructor(clauses) {
        const quotedClauses = clauses.map(terms => terms.map(quoteTerm).join(' '));
        if (quotedClauses.length === 1) {
            this.logPatternString = quotedClauses[0];
        }
        else {
            this.logPatternString = quotedClauses.map(alt => '?' + alt).join(' ');
        }
    }
}
/**
 * A string comparison for JSON values
 */
class JSONStringPattern extends JsonPattern {
    constructor(jsonField, comparison, value) {
        comparison = validateStringOperator(comparison);
        super(`${jsonField} ${comparison} ${quoteTerm(value)}`);
    }
}
/**
 * A number comparison for JSON values
 */
class JSONNumberPattern extends JsonPattern {
    constructor(jsonField, comparison, value) {
        comparison = validateNumericalOperator(comparison);
        super(`${jsonField} ${comparison} ${value}`);
    }
}
/**
 * A postfix operator for JSON patterns
 */
class JSONPostfixPattern extends JsonPattern {
    constructor(jsonField, postfix) {
        // No validation, we assume these are generated by trusted factory functions
        super(`${jsonField} ${postfix}`);
    }
}
/**
 * Combines multiple other JSON patterns with an operator
 */
class JSONAggregatePattern extends JsonPattern {
    constructor(operator, patterns) {
        if (operator !== '&&' && operator !== '||') {
            throw new Error('Operator must be one of && or ||');
        }
        const clauses = patterns.map(p => '(' + p.jsonPatternString + ')');
        super(clauses.join(` ${operator} `));
    }
}
const COL_ELLIPSIS = '...';
/**
 * Space delimited text pattern
 */
class SpaceDelimitedTextPattern {
    // TODO: Temporarily changed from private to protected to unblock build. We need to think
    //     about how to handle jsii types with private constructors.
    constructor(columns, restrictions) {
        this.columns = columns;
        this.restrictions = restrictions;
        // Private constructor so we validate in the .construct() factory function
    }
    /**
     * Construct a new instance of a space delimited text pattern
     *
     * Since this class must be public, we can't rely on the user only creating it through
     * the `LogPattern.spaceDelimited()` factory function. We must therefore validate the
     * argument in the constructor. Since we're returning a copy on every mutation, and we
     * don't want to re-validate the same things on every construction, we provide a limited
     * set of mutator functions and only validate the new data every time.
     */
    static construct(columns) {
        // Validation happens here because a user could instantiate this object directly without
        // going through the factory
        for (const column of columns) {
            if (!validColumnName(column)) {
                throw new Error(`Invalid column name: ${column}`);
            }
        }
        if (sum(columns.map(c => c === COL_ELLIPSIS ? 1 : 0)) > 1) {
            throw new Error("Can use at most one '...' column");
        }
        return new SpaceDelimitedTextPattern(columns, {});
    }
    /**
     * Restrict where the pattern applies
     */
    whereString(columnName, comparison, value) {
        if (columnName === COL_ELLIPSIS) {
            throw new Error("Can't use '...' in a restriction");
        }
        if (this.columns.indexOf(columnName) === -1) {
            throw new Error(`Column in restrictions that is not in columns: ${columnName}`);
        }
        comparison = validateStringOperator(comparison);
        return new SpaceDelimitedTextPattern(this.columns, this.addRestriction(columnName, {
            comparison,
            stringValue: value
        }));
    }
    /**
     * Restrict where the pattern applies
     */
    whereNumber(columnName, comparison, value) {
        if (columnName === COL_ELLIPSIS) {
            throw new Error("Can't use '...' in a restriction");
        }
        if (this.columns.indexOf(columnName) === -1) {
            throw new Error(`Column in restrictions that is not in columns: ${columnName}`);
        }
        comparison = validateNumericalOperator(comparison);
        return new SpaceDelimitedTextPattern(this.columns, this.addRestriction(columnName, {
            comparison,
            numberValue: value
        }));
    }
    get logPatternString() {
        return '[' + this.columns.map(this.columnExpression.bind(this)).join(', ') + ']';
    }
    /**
     * Return the column expression for the given column
     */
    columnExpression(column) {
        const restrictions = this.restrictions[column];
        if (!restrictions) {
            return column;
        }
        return restrictions.map(r => renderRestriction(column, r)).join(' && ');
    }
    /**
     * Make a copy of the current restrictions and add one
     */
    addRestriction(columnName, restriction) {
        const ret = {};
        for (const key of Object.keys(this.restrictions)) {
            ret[key] = this.restrictions[key].slice();
        }
        if (!(columnName in ret)) {
            ret[columnName] = [];
        }
        ret[columnName].push(restriction);
        return ret;
    }
}
exports.SpaceDelimitedTextPattern = SpaceDelimitedTextPattern;
/**
 * Quote a term for use in a pattern expression
 *
 * It's never wrong to quote a string term, and required if the term
 * contains non-alphanumerical characters, so we just always do it.
 *
 * Inner double quotes are escaped using a backslash.
 */
function quoteTerm(term) {
    return '"' + term.replace(/\\/g, '\\\\').replace(/"/g, '\\"') + '"';
}
/**
 * Return whether the given column name is valid in a space-delimited table
 */
function validColumnName(column) {
    return column === COL_ELLIPSIS || /^[a-zA-Z0-9_-]+$/.exec(column);
}
/**
 * Validate and normalize the string comparison operator
 *
 * Correct for a common typo/confusion, treat '==' as '='
 */
function validateStringOperator(operator) {
    if (operator === '==') {
        operator = '=';
    }
    if (operator !== '=' && operator !== '!=') {
        throw new Error(`Invalid comparison operator ('${operator}'), must be either '=' or '!='`);
    }
    return operator;
}
const VALID_OPERATORS = ['=', '!=', '<', '<=', '>', '>='];
/**
 * Validate and normalize numerical comparison operators
 *
 * Correct for a common typo/confusion, treat '==' as '='
 */
function validateNumericalOperator(operator) {
    // Correct for a common typo, treat '==' as '='
    if (operator === '==') {
        operator = '=';
    }
    if (VALID_OPERATORS.indexOf(operator) === -1) {
        throw new Error(`Invalid comparison operator ('${operator}'), must be one of ${VALID_OPERATORS.join(', ')}`);
    }
    return operator;
}
/**
 * Render a table restriction
 */
function renderRestriction(column, restriction) {
    if (restriction.numberValue !== undefined) {
        return `${column} ${restriction.comparison} ${restriction.numberValue}`;
    }
    else if (restriction.stringValue) {
        return `${column} ${restriction.comparison} ${quoteTerm(restriction.stringValue)}`;
    }
    else {
        throw new Error('Invalid restriction');
    }
}
function sum(xs) {
    return xs.reduce((a, c) => a + c, 0);
}
//# sourceMappingURL=data:application/json;base64,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