"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @stability stable
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        /** Set to keep track of what target accounts and regions we've already created event buses for */
        this._xEnvTargetsAdded = new Set();
        jsiiDeprecationWarnings._aws_cdk_aws_events_RuleProps(props);
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.any({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.any({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     * @stability stable
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).splitArn(eventRuleArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new targets.
     *
     * No-op if target is undefined.
     *
     * @stability stable
     */
    addTarget(target) {
        var _b, _c, _d;
        jsiiDeprecationWarnings._aws_cdk_aws_events_IRuleTarget(target);
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = (_b = targetProps.role) === null || _b === void 0 ? void 0 : _b.roleArn;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = ((_c = targetProps.targetResource.env) === null || _c === void 0 ? void 0 : _c.account) || targetStack.account;
            const targetRegion = ((_d = targetProps.targetResource.env) === null || _d === void 0 ? void 0 : _d.region) || targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            // if the target is in a different account or region and is defined in this CDK App
            // we can generate all the needed components:
            // - forwarding rule in the source stack (target: default event bus of the receiver region)
            // - eventbus permissions policy (creating an extra stack)
            // - receiver rule in the target stack (target: the actual target)
            if (!util_1.sameEnvDimension(sourceAccount, targetAccount) || !util_1.sameEnvDimension(sourceRegion, targetRegion)) {
                // cross-account and/or cross-region event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cross-account.html
                // for cross-account or cross-region events, we require a concrete target account and region
                if (!targetAccount || core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account or cross-region events');
                }
                if (!targetRegion || core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account or cross-region events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account or cross-region events');
                }
                // Don't exactly understand why this code was here (seems unlikely this rule would be violated), but
                // let's leave it in nonetheless.
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account or cross-region targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account or cross-region event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                // The target of this Rule will be the default event bus of the target environment
                this.ensureXEnvTargetEventBus(targetStack, targetAccount, targetRegion, id);
                // The actual rule lives in the target stack. Other than the account, it's identical to this one,
                // but only evaluated at render time (via a special subclass).
                //
                // FIXME: the MirrorRule is a bit silly, forwarding the exact same event to another event bus
                // and trigger on it there (there will be issues with construct references, for example). Especially
                // in the case of scheduled events, we will just trigger both rules in parallel in both environments.
                //
                // A better solution would be to have the source rule add a unique token to the the event,
                // and have the mirror rule trigger on that token only (thereby properly separating triggering, which
                // happens in the source env; and activating, which happens in the target env).
                //
                // Don't have time to do that right now.
                const mirrorRuleScope = this.obtainMirrorRuleScope(targetStack, targetAccount, targetRegion);
                new MirrorRule(mirrorRuleScope, `${core_1.Names.uniqueId(this)}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                }, this);
                return;
            }
        }
        // Here only if the target does not have a targetResource defined.
        // In such case we don't have to generate any extra component.
        // Note that this can also be an imported resource (i.e: EventBus target)
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            deadLetterConfig: targetProps.deadLetterConfig,
            retryPolicy: targetProps.retryPolicy,
            sqsParameters: targetProps.sqsParameters,
            httpParameters: targetProps.httpParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: (inputProps === null || inputProps === void 0 ? void 0 : inputProps.inputTemplate) !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule.
     *
     * If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *     {
     *       "resources": [ "r1" ],
     *       "detail": {
     *         "hello": [ 1 ]
     *       }
     *     }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *     {
     *       "resources": [ "r2" ],
     *       "detail": {
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * The resulting event pattern will be:
     *
     *     {
     *       "resources": [ "r1", "r2" ],
     *       "detail": {
     *         "hello": [ 1 ],
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * @stability stable
     */
    addEventPattern(eventPattern) {
        jsiiDeprecationWarnings._aws_cdk_aws_events_EventPattern(eventPattern);
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        return util_1.renderEventPattern(this.eventPattern);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
    /**
     * Make sure we add the target environments event bus as a target, and the target has permissions set up to receive our events
     *
     * For cross-account rules, uses a support stack to set up a policy on the target event bus.
     */
    ensureXEnvTargetEventBus(targetStack, targetAccount, targetRegion, id) {
        // the _actual_ target is just the event bus of the target's account
        // make sure we only add it once per account per region
        const key = `${targetAccount}:${targetRegion}`;
        if (this._xEnvTargetsAdded.has(key)) {
            return;
        }
        this._xEnvTargetsAdded.add(key);
        const eventBusArn = targetStack.formatArn({
            service: 'events',
            resource: 'event-bus',
            resourceName: 'default',
            region: targetRegion,
            account: targetAccount,
        });
        // For some reason, cross-region requires a Role (with `PutEvents` on the
        // target event bus) while cross-account doesn't
        const roleArn = !util_1.sameEnvDimension(targetRegion, core_1.Stack.of(this).region)
            ? this.crossRegionPutEventsRole(eventBusArn).roleArn
            : undefined;
        this.targets.push({
            id,
            arn: eventBusArn,
            roleArn,
        });
        // Add a policy to the target Event Bus to allow the source account/region to publish into it.
        //
        // Since this Event Bus permission needs to be deployed before the stack containing the Rule is deployed
        // (as EventBridge verifies whether you have permissions to the targets on rule creation), this needs
        // to be in a support stack.
        const sourceApp = this.node.root;
        const sourceAccount = core_1.Stack.of(this).account;
        // If different accounts, we need to add the permissions to the target eventbus
        //
        // For different region, no need for a policy on the target event bus (but a need
        // for a role).
        if (!util_1.sameEnvDimension(sourceAccount, targetAccount)) {
            const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
            let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
            if (!eventBusPolicyStack) {
                eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                    env: {
                        account: targetAccount,
                        region: targetRegion,
                    },
                    // The region in the stack name is rather redundant (it will always be the target region)
                    // Leaving it in for backwards compatibility.
                    stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                });
                new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                    action: 'events:PutEvents',
                    statementId: `Allow-account-${sourceAccount}`,
                    principal: sourceAccount,
                });
            }
            // deploy the event bus permissions before the source stack
            core_1.Stack.of(this).addDependency(eventBusPolicyStack);
        }
    }
    /**
     * Return the scope where the mirror rule should be created for x-env event targets
     *
     * This is the target resource's containing stack if it shares the same region (owned
     * resources), or should be a fresh support stack for imported resources.
     *
     * We don't implement the second yet, as I have to think long and hard on whether we
     * can reuse the existing support stack or not, and I don't have time for that right now.
     */
    obtainMirrorRuleScope(targetStack, targetAccount, targetRegion) {
        // for cross-account or cross-region events, we cannot create new components for an imported resource
        // because we don't have the target stack
        if (util_1.sameEnvDimension(targetStack.account, targetAccount) && util_1.sameEnvDimension(targetStack.region, targetRegion)) {
            return targetStack;
        }
        // For now, we don't do the work for the support stack yet
        throw new Error('Cannot create a cross-account or cross-region rule for an imported resource (create a stack with the right environment for the imported resource)');
    }
    /**
     * Obtain the Role for the EventBridge event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * events have the same target, they will share a role.
     * @internal
     */
    crossRegionPutEventsRole(eventBusArn) {
        const id = 'EventsRole';
        let role = this.node.tryFindChild(id);
        if (!role) {
            role = new aws_iam_1.Role(this, id, {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new aws_iam_1.ServicePrincipal('events.amazonaws.com'),
            });
        }
        role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [eventBusArn],
        }));
        return role;
    }
}
exports.Rule = Rule;
_a = JSII_RTTI_SYMBOL_1;
Rule[_a] = { fqn: "@aws-cdk/aws-events.Rule", version: "1.135.0" };
/**
 * A rule that mirrors another rule
 */
class MirrorRule extends Rule {
    constructor(scope, id, props, source) {
        super(scope, id, props);
        this.source = source;
    }
    _renderEventPattern() {
        return this.source._renderEventPattern();
    }
    /**
     * Override validate to be a no-op
     *
     * The rules are never stored on this object so there's nothing to validate.
     *
     * Instead, we mirror the other rule at render time.
     */
    validate() {
        return [];
    }
}
//# sourceMappingURL=data:application/json;base64,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