"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Macie = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [macie](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacieclassic.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Macie extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [macie](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacieclassic.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'macie';
        this.actionList = {
            "AssociateMemberAccount": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateMemberAccount.html",
                "description": "Enables the user to associate a specified AWS account with Amazon Macie as a member account.",
                "accessLevel": "Write"
            },
            "AssociateS3Resources": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateS3Resources.html",
                "description": "Enables the user to associate specified S3 resources with Amazon Macie for monitoring and data classification.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:SourceArn"
                ]
            },
            "DisassociateMemberAccount": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateMemberAccount.html",
                "description": "Enables the user to remove the specified member account from Amazon Macie.",
                "accessLevel": "Write"
            },
            "DisassociateS3Resources": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateS3Resources.html",
                "description": "Enables the user to remove specified S3 resources from being monitored by Amazon Macie.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:SourceArn"
                ]
            },
            "ListMemberAccounts": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListMemberAccounts.html",
                "description": "Enables the user to list all Amazon Macie member accounts for the current Macie master account.",
                "accessLevel": "List"
            },
            "ListS3Resources": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListS3Resources.html",
                "description": "Enables the user to list all the S3 resources associated with Amazon Macie.",
                "accessLevel": "List"
            },
            "UpdateS3Resources": {
                "url": "https://docs.aws.amazon.com/macie/1.0/APIReference/API_UpdateS3Resources.html",
                "description": "Enables the user to update the classification types for the specified S3 resources.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:SourceArn"
                ]
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Enables the user to associate a specified AWS account with Amazon Macie as a member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateMemberAccount.html
     */
    associateMemberAccount() {
        this.add('macie:AssociateMemberAccount');
        return this;
    }
    /**
     * Enables the user to associate specified S3 resources with Amazon Macie for monitoring and data classification.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateS3Resources.html
     */
    associateS3Resources() {
        this.add('macie:AssociateS3Resources');
        return this;
    }
    /**
     * Enables the user to remove the specified member account from Amazon Macie.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateMemberAccount.html
     */
    disassociateMemberAccount() {
        this.add('macie:DisassociateMemberAccount');
        return this;
    }
    /**
     * Enables the user to remove specified S3 resources from being monitored by Amazon Macie.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateS3Resources.html
     */
    disassociateS3Resources() {
        this.add('macie:DisassociateS3Resources');
        return this;
    }
    /**
     * Enables the user to list all Amazon Macie member accounts for the current Macie master account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListMemberAccounts.html
     */
    listMemberAccounts() {
        this.add('macie:ListMemberAccounts');
        return this;
    }
    /**
     * Enables the user to list all the S3 resources associated with Amazon Macie.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListS3Resources.html
     */
    listS3Resources() {
        this.add('macie:ListS3Resources');
        return this;
    }
    /**
     * Enables the user to update the classification types for the specified S3 resources.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_UpdateS3Resources.html
     */
    updateS3Resources() {
        this.add('macie:UpdateS3Resources');
        return this;
    }
    /**
     * Allow access to the specified actions only when the request operates on the specified aws resource
     *
     * Applies to actions:
     * - .associateS3Resources()
     * - .disassociateS3Resources()
     * - .updateS3Resources()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAwsSourceArn(value, operator) {
        return this.if(`aws:SourceArn`, value, operator || 'ArnEquals');
    }
}
exports.Macie = Macie;
//# sourceMappingURL=data:application/json;base64,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