"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Polly = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [polly](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpolly.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Polly extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [polly](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpolly.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'polly';
        this.actionList = {
            "DeleteLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html",
                "description": "Deletes the specified pronunciation lexicon stored in an AWS Region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "lexicon": {
                        "required": true
                    }
                }
            },
            "DescribeVoices": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html",
                "description": "Returns the list of voices that are available for use when requesting speech synthesis.",
                "accessLevel": "List"
            },
            "GetLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html",
                "description": "Returns the content of the specified pronunciation lexicon stored in an AWS Region.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "lexicon": {
                        "required": true
                    }
                }
            },
            "GetSpeechSynthesisTask": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html",
                "description": "Enables the user to get information about specific speech synthesis task.",
                "accessLevel": "Read"
            },
            "ListLexicons": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html",
                "description": "Returns a list of pronunciation lexicons stored in an AWS Region.",
                "accessLevel": "List"
            },
            "ListSpeechSynthesisTasks": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html",
                "description": "Enables the user to list requested speech synthesis tasks.",
                "accessLevel": "List"
            },
            "PutLexicon": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html",
                "description": "Stores a pronunciation lexicon in an AWS Region.",
                "accessLevel": "Write"
            },
            "StartSpeechSynthesisTask": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html",
                "description": "Enables the user to synthesize long inputs to provided S3 location.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:PutObject"
                ],
                "resourceTypes": {
                    "lexicon": {
                        "required": false
                    }
                }
            },
            "SynthesizeSpeech": {
                "url": "https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html",
                "description": "Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "lexicon": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "lexicon": {
                "name": "lexicon",
                "url": "https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html",
                "arn": "arn:${Partition}:polly:${Region}:${Account}:lexicon/${LexiconName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Deletes the specified pronunciation lexicon stored in an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html
     */
    deleteLexicon() {
        this.add('polly:DeleteLexicon');
        return this;
    }
    /**
     * Returns the list of voices that are available for use when requesting speech synthesis.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html
     */
    describeVoices() {
        this.add('polly:DescribeVoices');
        return this;
    }
    /**
     * Returns the content of the specified pronunciation lexicon stored in an AWS Region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html
     */
    getLexicon() {
        this.add('polly:GetLexicon');
        return this;
    }
    /**
     * Enables the user to get information about specific speech synthesis task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html
     */
    getSpeechSynthesisTask() {
        this.add('polly:GetSpeechSynthesisTask');
        return this;
    }
    /**
     * Returns a list of pronunciation lexicons stored in an AWS Region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html
     */
    listLexicons() {
        this.add('polly:ListLexicons');
        return this;
    }
    /**
     * Enables the user to list requested speech synthesis tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html
     */
    listSpeechSynthesisTasks() {
        this.add('polly:ListSpeechSynthesisTasks');
        return this;
    }
    /**
     * Stores a pronunciation lexicon in an AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html
     */
    putLexicon() {
        this.add('polly:PutLexicon');
        return this;
    }
    /**
     * Enables the user to synthesize long inputs to provided S3 location.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html
     */
    startSpeechSynthesisTask() {
        this.add('polly:StartSpeechSynthesisTask');
        return this;
    }
    /**
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html
     */
    synthesizeSpeech() {
        this.add('polly:SynthesizeSpeech');
        return this;
    }
    /**
     * Adds a resource of type lexicon to the statement
     *
     * https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html
     *
     * @param lexiconName - Identifier for the lexiconName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLexicon(lexiconName, account, region, partition) {
        var arn = 'arn:${Partition}:polly:${Region}:${Account}:lexicon/${LexiconName}';
        arn = arn.replace('${LexiconName}', lexiconName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Polly = Polly;
//# sourceMappingURL=data:application/json;base64,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