"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rekognition = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [rekognition](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rekognition extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [rekognition](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'rekognition';
        this.actionList = {
            "CompareFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CompareFaces.html",
                "description": "Compares a face in source input image with each face detected in the target input image.",
                "accessLevel": "Read"
            },
            "CreateCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateCollection.html",
                "description": "Creates a collection in an AWS region. You can then add faces to the collection using the IndexFaces API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProject.html",
                "description": "Creates a new Amazon Rekognition Custom Labels project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "CreateProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProjectVersion.html",
                "description": "Creates a new version of a model and begins training.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    },
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "CreateStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateStreamProcessor.html",
                "description": "Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming video.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    },
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DeleteCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteCollection.html",
                "description": "Deletes the specified collection. Note that this operation removes all faces in the collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DeleteFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteFaces.html",
                "description": "Deletes faces from a collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProject.html",
                "description": "Deletes a project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProjectVersion.html",
                "description": "Deletes a model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "DeleteStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteStreamProcessor.html",
                "description": "Deletes the stream processor identified by Name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DescribeCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeCollection.html",
                "description": "Describes the specified collection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DescribeProjectVersions": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjectVersions.html",
                "description": "Lists and describes the model versions in an Amazon Rekognition Custom Labels project.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeProjects": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjects.html",
                "description": "Lists and gets information about your Amazon Rekognition Custom Labels projects.",
                "accessLevel": "Read"
            },
            "DescribeStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeStreamProcessorh.html",
                "description": "Provides information about a stream processor created by CreateStreamProcessor.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DetectCustomLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectCustomLabels.html",
                "description": "Detects custom labels in a supplied image by using an Amazon Rekognition Custom Labels model version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "DetectFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectFaces.html",
                "description": "Detects human faces within an image (JPEG or PNG) provided as input.",
                "accessLevel": "Read"
            },
            "DetectLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectLabels.html",
                "description": "Detects instances of real-world labels within an image (JPEG or PNG) provided as input.",
                "accessLevel": "Read"
            },
            "DetectModerationLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectModerationLabels.html",
                "description": "Detects moderation labels within input image.",
                "accessLevel": "Read"
            },
            "DetectText": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectText.html",
                "description": "Detects text in the input image and converts it into machine-readable text.",
                "accessLevel": "Read"
            },
            "GetCelebrityInfo": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityInfo.html",
                "description": "Gets the name and additional information about a celebrity based on his or her Rekognition ID.",
                "accessLevel": "Read"
            },
            "GetCelebrityRecognition": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityRecognition.html",
                "description": "Gets the celebrity recognition results for a Rekognition Video analysis started by StartCelebrityRecognition.",
                "accessLevel": "Read"
            },
            "GetContentModeration": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetContentModeration.html",
                "description": "Gets the content moderation analysis results for a Rekognition Video analysis started by StartContentModeration.",
                "accessLevel": "Read"
            },
            "GetFaceDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceDetection.html",
                "description": "Gets face detection results for a Rekognition Video analysis started by StartFaceDetection.",
                "accessLevel": "Read"
            },
            "GetFaceSearch": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceSearch.html",
                "description": "Gets the face search results for Rekognition Video face search started by StartFaceSearch.",
                "accessLevel": "Read"
            },
            "GetLabelDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetLabelDetection.html",
                "description": "Gets the label detection results of a Rekognition Video analysis started by StartLabelDetection.",
                "accessLevel": "Read"
            },
            "GetPersonTracking": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetPersonTracking.html",
                "description": "Gets information about people detected within a video.",
                "accessLevel": "Read"
            },
            "GetSegmentDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetSegmentDetection.html",
                "description": "Gets segment detection results for a Rekognition Video analysis started by StartSegmentDetection.",
                "accessLevel": "Read"
            },
            "GetTextDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetTextDetection.html",
                "description": "Gets text detection results for a Rekognition Video analysis started by StartTextDetection.",
                "accessLevel": "Read"
            },
            "IndexFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_IndexFaces.html",
                "description": "Detects faces in the input image and adds them to the specified collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListCollections": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListCollections.html",
                "description": "Returns a list of collection IDs in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListFaces.html",
                "description": "Returns metadata for faces in the specified collection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListStreamProcessors": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListStreamProcessors.html",
                "description": "Gets a list of stream processors that you have created with CreateStreamProcessor.",
                "accessLevel": "List",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "RecognizeCelebrities": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_RecognizeCelebrities.html",
                "description": "Returns an array of celebrities recognized in the input image.",
                "accessLevel": "Read"
            },
            "SearchFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFaces.html",
                "description": "For a given input face ID, searches the specified collection for matching faces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "SearchFacesByImage": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFacesByImage.html",
                "description": "For a given input image, first detects the largest face in the image, and then searches the specified collection for matching faces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "StartCelebrityRecognition": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartCelebrityRecognition.html",
                "description": "Starts asynchronous recognition of celebrities in a video.",
                "accessLevel": "Write"
            },
            "StartContentModeration": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartContentModeration.html",
                "description": "Starts asynchronous detection of explicit or suggestive adult content in a video.",
                "accessLevel": "Write"
            },
            "StartFaceDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceDetection.html",
                "description": "Starts asynchronous detection of faces in a video.",
                "accessLevel": "Write"
            },
            "StartFaceSearch": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceSearch.html",
                "description": "Starts the asynchronous search for faces in a collection that match the faces of persons detected in a video.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "StartLabelDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartLabelDetection.html",
                "description": "Starts asynchronous detection of labels in a video.",
                "accessLevel": "Write"
            },
            "StartPersonTracking": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartPersonTracking.html",
                "description": "Starts the asynchronous tracking of persons in a video.",
                "accessLevel": "Write"
            },
            "StartProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartProjectVersion.html",
                "description": "Starts the deployment of a model version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "StartSegmentDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartSegmentDetection.html",
                "description": "Starts asynchronous detection of segments in a video.",
                "accessLevel": "Write"
            },
            "StartStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartStreamProcessor.html",
                "description": "Starts processing a stream processor.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "StartTextDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartTextDetection.html",
                "description": "Starts asynchronous detection of text in a video.",
                "accessLevel": "Write"
            },
            "StopProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StopProjectVersion.html",
                "description": "Stops a deployed model version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "StopStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StopStreamProcessor.html",
                "description": "Stops a running stream processor that was created by CreateStreamProcessor.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "collection": {
                "name": "collection",
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/howitworks-collection.html",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:collection/${CollectionId}",
                "conditionKeys": []
            },
            "streamprocessor": {
                "name": "streamprocessor",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:streamprocessor/${StreamprocessorId}",
                "conditionKeys": []
            },
            "project": {
                "name": "project",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/${CreationTimestamp}",
                "conditionKeys": []
            },
            "projectversion": {
                "name": "projectversion",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/version/${VersionName}/${CreationTimestamp}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Compares a face in source input image with each face detected in the target input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CompareFaces.html
     */
    compareFaces() {
        this.add('rekognition:CompareFaces');
        return this;
    }
    /**
     * Creates a collection in an AWS region. You can then add faces to the collection using the IndexFaces API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateCollection.html
     */
    createCollection() {
        this.add('rekognition:CreateCollection');
        return this;
    }
    /**
     * Creates a new Amazon Rekognition Custom Labels project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProject.html
     */
    createProject() {
        this.add('rekognition:CreateProject');
        return this;
    }
    /**
     * Creates a new version of a model and begins training.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProjectVersion.html
     */
    createProjectVersion() {
        this.add('rekognition:CreateProjectVersion');
        return this;
    }
    /**
     * Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateStreamProcessor.html
     */
    createStreamProcessor() {
        this.add('rekognition:CreateStreamProcessor');
        return this;
    }
    /**
     * Deletes the specified collection. Note that this operation removes all faces in the collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteCollection.html
     */
    deleteCollection() {
        this.add('rekognition:DeleteCollection');
        return this;
    }
    /**
     * Deletes faces from a collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteFaces.html
     */
    deleteFaces() {
        this.add('rekognition:DeleteFaces');
        return this;
    }
    /**
     * Deletes a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProject.html
     */
    deleteProject() {
        this.add('rekognition:DeleteProject');
        return this;
    }
    /**
     * Deletes a model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProjectVersion.html
     */
    deleteProjectVersion() {
        this.add('rekognition:DeleteProjectVersion');
        return this;
    }
    /**
     * Deletes the stream processor identified by Name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteStreamProcessor.html
     */
    deleteStreamProcessor() {
        this.add('rekognition:DeleteStreamProcessor');
        return this;
    }
    /**
     * Describes the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeCollection.html
     */
    describeCollection() {
        this.add('rekognition:DescribeCollection');
        return this;
    }
    /**
     * Lists and describes the model versions in an Amazon Rekognition Custom Labels project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjectVersions.html
     */
    describeProjectVersions() {
        this.add('rekognition:DescribeProjectVersions');
        return this;
    }
    /**
     * Lists and gets information about your Amazon Rekognition Custom Labels projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjects.html
     */
    describeProjects() {
        this.add('rekognition:DescribeProjects');
        return this;
    }
    /**
     * Provides information about a stream processor created by CreateStreamProcessor.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeStreamProcessorh.html
     */
    describeStreamProcessor() {
        this.add('rekognition:DescribeStreamProcessor');
        return this;
    }
    /**
     * Detects custom labels in a supplied image by using an Amazon Rekognition Custom Labels model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectCustomLabels.html
     */
    detectCustomLabels() {
        this.add('rekognition:DetectCustomLabels');
        return this;
    }
    /**
     * Detects human faces within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectFaces.html
     */
    detectFaces() {
        this.add('rekognition:DetectFaces');
        return this;
    }
    /**
     * Detects instances of real-world labels within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectLabels.html
     */
    detectLabels() {
        this.add('rekognition:DetectLabels');
        return this;
    }
    /**
     * Detects moderation labels within input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectModerationLabels.html
     */
    detectModerationLabels() {
        this.add('rekognition:DetectModerationLabels');
        return this;
    }
    /**
     * Detects text in the input image and converts it into machine-readable text.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectText.html
     */
    detectText() {
        this.add('rekognition:DetectText');
        return this;
    }
    /**
     * Gets the name and additional information about a celebrity based on his or her Rekognition ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityInfo.html
     */
    getCelebrityInfo() {
        this.add('rekognition:GetCelebrityInfo');
        return this;
    }
    /**
     * Gets the celebrity recognition results for a Rekognition Video analysis started by StartCelebrityRecognition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityRecognition.html
     */
    getCelebrityRecognition() {
        this.add('rekognition:GetCelebrityRecognition');
        return this;
    }
    /**
     * Gets the content moderation analysis results for a Rekognition Video analysis started by StartContentModeration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetContentModeration.html
     */
    getContentModeration() {
        this.add('rekognition:GetContentModeration');
        return this;
    }
    /**
     * Gets face detection results for a Rekognition Video analysis started by StartFaceDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceDetection.html
     */
    getFaceDetection() {
        this.add('rekognition:GetFaceDetection');
        return this;
    }
    /**
     * Gets the face search results for Rekognition Video face search started by StartFaceSearch.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceSearch.html
     */
    getFaceSearch() {
        this.add('rekognition:GetFaceSearch');
        return this;
    }
    /**
     * Gets the label detection results of a Rekognition Video analysis started by StartLabelDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetLabelDetection.html
     */
    getLabelDetection() {
        this.add('rekognition:GetLabelDetection');
        return this;
    }
    /**
     * Gets information about people detected within a video.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetPersonTracking.html
     */
    getPersonTracking() {
        this.add('rekognition:GetPersonTracking');
        return this;
    }
    /**
     * Gets segment detection results for a Rekognition Video analysis started by StartSegmentDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetSegmentDetection.html
     */
    getSegmentDetection() {
        this.add('rekognition:GetSegmentDetection');
        return this;
    }
    /**
     * Gets text detection results for a Rekognition Video analysis started by StartTextDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetTextDetection.html
     */
    getTextDetection() {
        this.add('rekognition:GetTextDetection');
        return this;
    }
    /**
     * Detects faces in the input image and adds them to the specified collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_IndexFaces.html
     */
    indexFaces() {
        this.add('rekognition:IndexFaces');
        return this;
    }
    /**
     * Returns a list of collection IDs in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListCollections.html
     */
    listCollections() {
        this.add('rekognition:ListCollections');
        return this;
    }
    /**
     * Returns metadata for faces in the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListFaces.html
     */
    listFaces() {
        this.add('rekognition:ListFaces');
        return this;
    }
    /**
     * Gets a list of stream processors that you have created with CreateStreamProcessor.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListStreamProcessors.html
     */
    listStreamProcessors() {
        this.add('rekognition:ListStreamProcessors');
        return this;
    }
    /**
     * Returns an array of celebrities recognized in the input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_RecognizeCelebrities.html
     */
    recognizeCelebrities() {
        this.add('rekognition:RecognizeCelebrities');
        return this;
    }
    /**
     * For a given input face ID, searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFaces.html
     */
    searchFaces() {
        this.add('rekognition:SearchFaces');
        return this;
    }
    /**
     * For a given input image, first detects the largest face in the image, and then searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFacesByImage.html
     */
    searchFacesByImage() {
        this.add('rekognition:SearchFacesByImage');
        return this;
    }
    /**
     * Starts asynchronous recognition of celebrities in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartCelebrityRecognition.html
     */
    startCelebrityRecognition() {
        this.add('rekognition:StartCelebrityRecognition');
        return this;
    }
    /**
     * Starts asynchronous detection of explicit or suggestive adult content in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartContentModeration.html
     */
    startContentModeration() {
        this.add('rekognition:StartContentModeration');
        return this;
    }
    /**
     * Starts asynchronous detection of faces in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceDetection.html
     */
    startFaceDetection() {
        this.add('rekognition:StartFaceDetection');
        return this;
    }
    /**
     * Starts the asynchronous search for faces in a collection that match the faces of persons detected in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceSearch.html
     */
    startFaceSearch() {
        this.add('rekognition:StartFaceSearch');
        return this;
    }
    /**
     * Starts asynchronous detection of labels in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartLabelDetection.html
     */
    startLabelDetection() {
        this.add('rekognition:StartLabelDetection');
        return this;
    }
    /**
     * Starts the asynchronous tracking of persons in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartPersonTracking.html
     */
    startPersonTracking() {
        this.add('rekognition:StartPersonTracking');
        return this;
    }
    /**
     * Starts the deployment of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartProjectVersion.html
     */
    startProjectVersion() {
        this.add('rekognition:StartProjectVersion');
        return this;
    }
    /**
     * Starts asynchronous detection of segments in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartSegmentDetection.html
     */
    startSegmentDetection() {
        this.add('rekognition:StartSegmentDetection');
        return this;
    }
    /**
     * Starts processing a stream processor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartStreamProcessor.html
     */
    startStreamProcessor() {
        this.add('rekognition:StartStreamProcessor');
        return this;
    }
    /**
     * Starts asynchronous detection of text in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartTextDetection.html
     */
    startTextDetection() {
        this.add('rekognition:StartTextDetection');
        return this;
    }
    /**
     * Stops a deployed model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopProjectVersion.html
     */
    stopProjectVersion() {
        this.add('rekognition:StopProjectVersion');
        return this;
    }
    /**
     * Stops a running stream processor that was created by CreateStreamProcessor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopStreamProcessor.html
     */
    stopStreamProcessor() {
        this.add('rekognition:StopStreamProcessor');
        return this;
    }
    /**
     * Adds a resource of type collection to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/howitworks-collection.html
     *
     * @param collectionId - Identifier for the collectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCollection(collectionId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:collection/${CollectionId}';
        arn = arn.replace('${CollectionId}', collectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type streamprocessor to the statement
     *
     * @param streamprocessorId - Identifier for the streamprocessorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStreamprocessor(streamprocessorId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:streamprocessor/${StreamprocessorId}';
        arn = arn.replace('${StreamprocessorId}', streamprocessorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type projectversion to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param versionName - Identifier for the versionName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProjectversion(projectName, versionName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/version/${VersionName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${VersionName}', versionName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Rekognition = Rekognition;
//# sourceMappingURL=data:application/json;base64,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