"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Worklink = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [worklink](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworklink.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Worklink extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [worklink](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworklink.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'worklink';
        this.actionList = {
            "AssociateDomain": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_AssociateDomain.html",
                "description": "Grants permission to associate a domain with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "AssociateWebsiteAuthorizationProvider": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteAuthorizationProvider.html",
                "description": "Grants permission to associate a website authorization provider with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "AssociateWebsiteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteCertificateAuthority.html",
                "description": "Grants permission to associate a website certificate authority with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_CreateFleet.html",
                "description": "Grants permission to create an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteFleet": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DeleteFleet.html",
                "description": "Grants permission to delete an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeAuditStreamConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeAuditStreamConfiguration.html",
                "description": "Grants permission to describe the audit stream configuration for an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeCompanyNetworkConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeCompanyNetworkConfiguration.html",
                "description": "Grants permission to describe the company network configuration for an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeDevice": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevice.html",
                "description": "Grants permission to describe details of a device associated with an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeDevicePolicyConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevicePolicyConfiguration.html",
                "description": "Grants permission to describe the device policy configuration for an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDomain.html",
                "description": "Grants permission to describe details about a domain associated with an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeFleetMetadata": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeFleetMetadata.html",
                "description": "Grants permission to describe metadata of an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeIdentityProviderConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeIdentityProviderConfiguration.html",
                "description": "Grants permission to describe the identity provider configuration for an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeWebsiteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeWebsiteCertificateAuthority.html",
                "description": "Grants permission to describe a website certificate authority associated with an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DisassociateDomain": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateDomain.html",
                "description": "Grants permission to disassociate a domain from an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DisassociateWebsiteAuthorizationProvider": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteAuthorizationProvider.html",
                "description": "Grants permission to disassociate a website authorization provider from an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DisassociateWebsiteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteCertificateAuthority.html",
                "description": "Grants permission to disassociate a website certificate authority from an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListDevices": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListDevices.html",
                "description": "Grants permission to list the devices associated with an Amazon WorkLink fleet",
                "accessLevel": "List",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListDomains.html",
                "description": "Grants permission to list the associated domains for an Amazon WorkLink fleet",
                "accessLevel": "List",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListFleets": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListFleets.html",
                "description": "Grants permission to list the Amazon WorkLink fleets associated with the account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for a resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListWebsiteAuthorizationProviders": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteAuthorizationProviders.html",
                "description": "Grants permission to list the website authorization providers for an Amazon WorkLink fleet",
                "accessLevel": "List",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListWebsiteCertificateAuthorities": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteCertificateAuthorities.html",
                "description": "Grants permission to list the website certificate authorities associated with an Amazon WorkLink fleet",
                "accessLevel": "List",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "RestoreDomainAccess": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_RestoreDomainAccess.html",
                "description": "Grants permission to restore access to a domain associated with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "RevokeDomainAccess": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_RevokeDomainAccess.html",
                "description": "Grants permission to revoke access to a domain associated with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "SignOutUser": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_SignOutUser.html",
                "description": "Grants permission to sign out a user from an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_TagResource.html",
                "description": "Grants permission to add one or more tags to a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UntagResource.html",
                "description": "Grants permission to remove one or more tags from a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAuditStreamConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateAuditStreamConfiguration.html",
                "description": "Grants permission to update the audit stream configuration for an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateCompanyNetworkConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateCompanyNetworkConfiguration.html",
                "description": "Grants permission to update the company network configuration for an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateDevicePolicyConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDevicePolicyConfiguration.html",
                "description": "Grants permission to update the device policy configuration for an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateDomainMetadata": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDomainMetadata.html",
                "description": "Grants permission to update the metadata for a domain associated with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateFleetMetadata": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateFleetMetadata.html",
                "description": "Grants permission to update the metadata of an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateIdentityProviderConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateIdentityProviderConfiguration.html",
                "description": "Grants permission to update the identity provider configuration for an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "fleet": {
                "name": "fleet",
                "url": "https://docs.aws.amazon.com/worklink/latest/api/worklink-resources.html#Fleet",
                "arn": "arn:${Partition}:worklink::${Account}:fleet/${FleetName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate a domain with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateDomain.html
     */
    associateDomain() {
        this.add('worklink:AssociateDomain');
        return this;
    }
    /**
     * Grants permission to associate a website authorization provider with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteAuthorizationProvider.html
     */
    associateWebsiteAuthorizationProvider() {
        this.add('worklink:AssociateWebsiteAuthorizationProvider');
        return this;
    }
    /**
     * Grants permission to associate a website certificate authority with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteCertificateAuthority.html
     */
    associateWebsiteCertificateAuthority() {
        this.add('worklink:AssociateWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to create an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_CreateFleet.html
     */
    createFleet() {
        this.add('worklink:CreateFleet');
        return this;
    }
    /**
     * Grants permission to delete an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DeleteFleet.html
     */
    deleteFleet() {
        this.add('worklink:DeleteFleet');
        return this;
    }
    /**
     * Grants permission to describe the audit stream configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeAuditStreamConfiguration.html
     */
    describeAuditStreamConfiguration() {
        this.add('worklink:DescribeAuditStreamConfiguration');
        return this;
    }
    /**
     * Grants permission to describe the company network configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeCompanyNetworkConfiguration.html
     */
    describeCompanyNetworkConfiguration() {
        this.add('worklink:DescribeCompanyNetworkConfiguration');
        return this;
    }
    /**
     * Grants permission to describe details of a device associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevice.html
     */
    describeDevice() {
        this.add('worklink:DescribeDevice');
        return this;
    }
    /**
     * Grants permission to describe the device policy configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevicePolicyConfiguration.html
     */
    describeDevicePolicyConfiguration() {
        this.add('worklink:DescribeDevicePolicyConfiguration');
        return this;
    }
    /**
     * Grants permission to describe details about a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDomain.html
     */
    describeDomain() {
        this.add('worklink:DescribeDomain');
        return this;
    }
    /**
     * Grants permission to describe metadata of an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeFleetMetadata.html
     */
    describeFleetMetadata() {
        this.add('worklink:DescribeFleetMetadata');
        return this;
    }
    /**
     * Grants permission to describe the identity provider configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeIdentityProviderConfiguration.html
     */
    describeIdentityProviderConfiguration() {
        this.add('worklink:DescribeIdentityProviderConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a website certificate authority associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeWebsiteCertificateAuthority.html
     */
    describeWebsiteCertificateAuthority() {
        this.add('worklink:DescribeWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to disassociate a domain from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateDomain.html
     */
    disassociateDomain() {
        this.add('worklink:DisassociateDomain');
        return this;
    }
    /**
     * Grants permission to disassociate a website authorization provider from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteAuthorizationProvider.html
     */
    disassociateWebsiteAuthorizationProvider() {
        this.add('worklink:DisassociateWebsiteAuthorizationProvider');
        return this;
    }
    /**
     * Grants permission to disassociate a website certificate authority from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteCertificateAuthority.html
     */
    disassociateWebsiteCertificateAuthority() {
        this.add('worklink:DisassociateWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to list the devices associated with an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListDevices.html
     */
    listDevices() {
        this.add('worklink:ListDevices');
        return this;
    }
    /**
     * Grants permission to list the associated domains for an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListDomains.html
     */
    listDomains() {
        this.add('worklink:ListDomains');
        return this;
    }
    /**
     * Grants permission to list the Amazon WorkLink fleets associated with the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListFleets.html
     */
    listFleets() {
        this.add('worklink:ListFleets');
        return this;
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('worklink:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the website authorization providers for an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteAuthorizationProviders.html
     */
    listWebsiteAuthorizationProviders() {
        this.add('worklink:ListWebsiteAuthorizationProviders');
        return this;
    }
    /**
     * Grants permission to list the website certificate authorities associated with an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteCertificateAuthorities.html
     */
    listWebsiteCertificateAuthorities() {
        this.add('worklink:ListWebsiteCertificateAuthorities');
        return this;
    }
    /**
     * Grants permission to restore access to a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_RestoreDomainAccess.html
     */
    restoreDomainAccess() {
        this.add('worklink:RestoreDomainAccess');
        return this;
    }
    /**
     * Grants permission to revoke access to a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_RevokeDomainAccess.html
     */
    revokeDomainAccess() {
        this.add('worklink:RevokeDomainAccess');
        return this;
    }
    /**
     * Grants permission to sign out a user from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_SignOutUser.html
     */
    signOutUser() {
        this.add('worklink:SignOutUser');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('worklink:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('worklink:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the audit stream configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateAuditStreamConfiguration.html
     */
    updateAuditStreamConfiguration() {
        this.add('worklink:UpdateAuditStreamConfiguration');
        return this;
    }
    /**
     * Grants permission to update the company network configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateCompanyNetworkConfiguration.html
     */
    updateCompanyNetworkConfiguration() {
        this.add('worklink:UpdateCompanyNetworkConfiguration');
        return this;
    }
    /**
     * Grants permission to update the device policy configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDevicePolicyConfiguration.html
     */
    updateDevicePolicyConfiguration() {
        this.add('worklink:UpdateDevicePolicyConfiguration');
        return this;
    }
    /**
     * Grants permission to update the metadata for a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDomainMetadata.html
     */
    updateDomainMetadata() {
        this.add('worklink:UpdateDomainMetadata');
        return this;
    }
    /**
     * Grants permission to update the metadata of an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateFleetMetadata.html
     */
    updateFleetMetadata() {
        this.add('worklink:UpdateFleetMetadata');
        return this;
    }
    /**
     * Grants permission to update the identity provider configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateIdentityProviderConfiguration.html
     */
    updateIdentityProviderConfiguration() {
        this.add('worklink:UpdateIdentityProviderConfiguration');
        return this;
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/worklink/latest/api/worklink-resources.html#Fleet
     *
     * @param fleetName - Identifier for the fleetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetName, account, partition) {
        var arn = 'arn:${Partition}:worklink::${Account}:fleet/${FleetName}';
        arn = arn.replace('${FleetName}', fleetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createFleet()
     * - .deleteFleet()
     * - .describeFleetMetadata()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - fleet
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createFleet()
     * - .deleteFleet()
     * - .describeFleetMetadata()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Worklink = Worklink;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9ud29ya2xpbmsuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbWF6b253b3JrbGluay50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBb0U7QUFFcEU7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLHdCQUFlO0lBa1czQzs7OztPQUlHO0lBQ0gsWUFBYSxHQUFZO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQXZXTixrQkFBYSxHQUFHLFVBQVUsQ0FBQztRQUN4QixlQUFVLEdBQVk7WUFDOUIsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGFBQWEsRUFBRSx1RUFBdUU7Z0JBQ3RGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUNBQXVDLEVBQUU7Z0JBQ3ZDLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSwrRkFBK0Y7Z0JBQzlHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsc0NBQXNDLEVBQUU7Z0JBQ3RDLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSw4RkFBOEY7Z0JBQzdHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxzRUFBc0U7Z0JBQzdFLGFBQWEsRUFBRSxzREFBc0Q7Z0JBQ3JFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHNFQUFzRTtnQkFDN0UsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGtDQUFrQyxFQUFFO2dCQUNsQyxLQUFLLEVBQUUsMkZBQTJGO2dCQUNsRyxhQUFhLEVBQUUsMkZBQTJGO2dCQUMxRyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHFDQUFxQyxFQUFFO2dCQUNyQyxLQUFLLEVBQUUsOEZBQThGO2dCQUNyRyxhQUFhLEVBQUUsOEZBQThGO2dCQUM3RyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUseUVBQXlFO2dCQUNoRixhQUFhLEVBQUUsNEZBQTRGO2dCQUMzRyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELG1DQUFtQyxFQUFFO2dCQUNuQyxLQUFLLEVBQUUsNEZBQTRGO2dCQUNuRyxhQUFhLEVBQUUsNEZBQTRGO2dCQUMzRyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUseUVBQXlFO2dCQUNoRixhQUFhLEVBQUUsK0ZBQStGO2dCQUM5RyxhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHVCQUF1QixFQUFFO2dCQUN2QixLQUFLLEVBQUUsZ0ZBQWdGO2dCQUN2RixhQUFhLEVBQUUsb0VBQW9FO2dCQUNuRixhQUFhLEVBQUUsTUFBTTtnQkFDckIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsdUNBQXVDLEVBQUU7Z0JBQ3ZDLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSxnR0FBZ0c7Z0JBQy9HLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUNBQXFDLEVBQUU7Z0JBQ3JDLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSx3R0FBd0c7Z0JBQ3ZILGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSwwRUFBMEU7Z0JBQ3pGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMENBQTBDLEVBQUU7Z0JBQzFDLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLGFBQWEsRUFBRSxrR0FBa0c7Z0JBQ2pILGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QseUNBQXlDLEVBQUU7Z0JBQ3pDLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSxpR0FBaUc7Z0JBQ2hILGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxzRUFBc0U7Z0JBQzdFLGFBQWEsRUFBRSxnRkFBZ0Y7Z0JBQy9GLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxzRUFBc0U7Z0JBQzdFLGFBQWEsRUFBRSwrRUFBK0U7Z0JBQzlGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSxxRUFBcUU7Z0JBQzVFLGFBQWEsRUFBRSxrRkFBa0Y7Z0JBQ2pHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGFBQWEsRUFBRSwrQ0FBK0M7Z0JBQzlELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUNBQW1DLEVBQUU7Z0JBQ25DLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSw0RkFBNEY7Z0JBQzNHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUNBQW1DLEVBQUU7Z0JBQ25DLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSx3R0FBd0c7Z0JBQ3ZILGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGFBQWEsRUFBRSwwRkFBMEY7Z0JBQ3pHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSx5RkFBeUY7Z0JBQ3hHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxzRUFBc0U7Z0JBQzdFLGFBQWEsRUFBRSxvRUFBb0U7Z0JBQ25GLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxzRUFBc0U7Z0JBQzdFLGFBQWEsRUFBRSx5REFBeUQ7Z0JBQ3hFLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLHdFQUF3RTtnQkFDL0UsYUFBYSxFQUFFLDhEQUE4RDtnQkFDN0UsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxnQ0FBZ0MsRUFBRTtnQkFDaEMsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHlGQUF5RjtnQkFDeEcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQ0FBbUMsRUFBRTtnQkFDbkMsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLDRGQUE0RjtnQkFDM0csYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxpQ0FBaUMsRUFBRTtnQkFDakMsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLDBGQUEwRjtnQkFDekcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLGdHQUFnRztnQkFDL0csYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHNFQUFzRTtnQkFDckYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQ0FBcUMsRUFBRTtnQkFDckMsS0FBSyxFQUFFLDhGQUE4RjtnQkFDckcsYUFBYSxFQUFFLDhGQUE4RjtnQkFDN0csYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDO1FBQ1Esa0JBQWEsR0FBa0I7WUFDdkMsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRSxPQUFPO2dCQUNmLEtBQUssRUFBRSwrRUFBK0U7Z0JBQ3RGLEtBQUssRUFBRSwwREFBMEQ7Z0JBQ2pFLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7U0FDRixDQUFDO0lBU0YsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxJQUFJLENBQUMsR0FBRyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7UUFDM0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLElBQUksQ0FBQyxHQUFHLENBQUMsK0NBQStDLENBQUMsQ0FBQztRQUMxRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdDQUFnQztRQUNyQyxJQUFJLENBQUMsR0FBRyxDQUFDLDJDQUEyQyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLElBQUksQ0FBQyxHQUFHLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN6RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxJQUFJLENBQUMsR0FBRyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7UUFDM0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLElBQUksQ0FBQyxHQUFHLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN6RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdDQUF3QztRQUM3QyxJQUFJLENBQUMsR0FBRyxDQUFDLG1EQUFtRCxDQUFDLENBQUM7UUFDOUQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUNBQXVDO1FBQzVDLElBQUksQ0FBQyxHQUFHLENBQUMsa0RBQWtELENBQUMsQ0FBQztRQUM3RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLElBQUksQ0FBQyxHQUFHLENBQUMsNENBQTRDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLElBQUksQ0FBQyxHQUFHLENBQUMseUNBQXlDLENBQUMsQ0FBQztRQUNwRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxJQUFJLENBQUMsR0FBRyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7UUFDckQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxJQUFJLENBQUMsR0FBRyxDQUFDLDhDQUE4QyxDQUFDLENBQUM7UUFDekQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxPQUFnQixFQUFFLFNBQWtCO1FBQ3BFLElBQUksR0FBRyxHQUFHLDBEQUEwRCxDQUFDO1FBQ3JFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM3QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUFpQjtRQUNoRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBaUI7UUFDakYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQzdELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0NBQ0Y7QUFuMUJELDRCQW0xQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFt3b3JrbGlua10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9ud29ya2xpbmsuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBXb3JrbGluayBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3dvcmtsaW5rJztcbiAgcHJvdGVjdGVkIGFjdGlvbkxpc3Q6IEFjdGlvbnMgPSB7XG4gICAgXCJBc3NvY2lhdGVEb21haW5cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfQXNzb2NpYXRlRG9tYWluLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgYSBkb21haW4gd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQXNzb2NpYXRlV2Vic2l0ZUF1dGhvcml6YXRpb25Qcm92aWRlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9Bc3NvY2lhdGVXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgYSB3ZWJzaXRlIGF1dGhvcml6YXRpb24gcHJvdmlkZXIgd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQXNzb2NpYXRlV2Vic2l0ZUNlcnRpZmljYXRlQXV0aG9yaXR5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Fzc29jaWF0ZVdlYnNpdGVDZXJ0aWZpY2F0ZUF1dGhvcml0eS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGEgd2Vic2l0ZSBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkgd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlRmxlZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlRmxlZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZUZsZWV0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0RlbGV0ZUZsZWV0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlc2NyaWJlQXVkaXRTdHJlYW1Db25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlQXVkaXRTdHJlYW1Db25maWd1cmF0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSB0aGUgYXVkaXQgc3RyZWFtIGNvbmZpZ3VyYXRpb24gZm9yIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlQ29tcGFueU5ldHdvcmtDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlQ29tcGFueU5ldHdvcmtDb25maWd1cmF0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSB0aGUgY29tcGFueSBuZXR3b3JrIGNvbmZpZ3VyYXRpb24gZm9yIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlRGV2aWNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlRGV2aWNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBkZXRhaWxzIG9mIGEgZGV2aWNlIGFzc29jaWF0ZWQgd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZURldmljZVBvbGljeUNvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVEZXZpY2VQb2xpY3lDb25maWd1cmF0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSB0aGUgZGV2aWNlIHBvbGljeSBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZURvbWFpblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9EZXNjcmliZURvbWFpbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgZGV0YWlscyBhYm91dCBhIGRvbWFpbiBhc3NvY2lhdGVkIHdpdGggYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVGbGVldE1ldGFkYXRhXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlRmxlZXRNZXRhZGF0YS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgbWV0YWRhdGEgb2YgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVJZGVudGl0eVByb3ZpZGVyQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9EZXNjcmliZUlkZW50aXR5UHJvdmlkZXJDb25maWd1cmF0aW9uLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSB0aGUgaWRlbnRpdHkgcHJvdmlkZXIgY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVXZWJzaXRlQ2VydGlmaWNhdGVBdXRob3JpdHlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVXZWJzaXRlQ2VydGlmaWNhdGVBdXRob3JpdHkuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGEgd2Vic2l0ZSBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkgYXNzb2NpYXRlZCB3aXRoIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRpc2Fzc29jaWF0ZURvbWFpblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9EaXNhc3NvY2lhdGVEb21haW4uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSBhIGRvbWFpbiBmcm9tIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEaXNhc3NvY2lhdGVXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Rpc2Fzc29jaWF0ZVdlYnNpdGVBdXRob3JpemF0aW9uUHJvdmlkZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSBhIHdlYnNpdGUgYXV0aG9yaXphdGlvbiBwcm92aWRlciBmcm9tIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEaXNhc3NvY2lhdGVXZWJzaXRlQ2VydGlmaWNhdGVBdXRob3JpdHlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGlzYXNzb2NpYXRlV2Vic2l0ZUNlcnRpZmljYXRlQXV0aG9yaXR5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYSB3ZWJzaXRlIGNlcnRpZmljYXRlIGF1dGhvcml0eSBmcm9tIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0RGV2aWNlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9MaXN0RGV2aWNlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgZGV2aWNlcyBhc3NvY2lhdGVkIHdpdGggYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdERvbWFpbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfTGlzdERvbWFpbnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGFzc29jaWF0ZWQgZG9tYWlucyBmb3IgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdEZsZWV0c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9MaXN0RmxlZXRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBBbWF6b24gV29ya0xpbmsgZmxlZXRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgYWNjb3VudFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0VGFnc0ZvclJlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGFncyBmb3IgYSByZXNvdXJjZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9MaXN0V2Vic2l0ZUF1dGhvcml6YXRpb25Qcm92aWRlcnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHdlYnNpdGUgYXV0aG9yaXphdGlvbiBwcm92aWRlcnMgZm9yIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RXZWJzaXRlQ2VydGlmaWNhdGVBdXRob3JpdGllc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9MaXN0V2Vic2l0ZUNlcnRpZmljYXRlQXV0aG9yaXRpZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHdlYnNpdGUgY2VydGlmaWNhdGUgYXV0aG9yaXRpZXMgYXNzb2NpYXRlZCB3aXRoIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlJlc3RvcmVEb21haW5BY2Nlc3NcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfUmVzdG9yZURvbWFpbkFjY2Vzcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVzdG9yZSBhY2Nlc3MgdG8gYSBkb21haW4gYXNzb2NpYXRlZCB3aXRoIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJSZXZva2VEb21haW5BY2Nlc3NcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfUmV2b2tlRG9tYWluQWNjZXNzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXZva2UgYWNjZXNzIHRvIGEgZG9tYWluIGFzc29jaWF0ZWQgd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU2lnbk91dFVzZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfU2lnbk91dFVzZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHNpZ24gb3V0IGEgdXNlciBmcm9tIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJUYWdSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9UYWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG9uZSBvciBtb3JlIHRhZ3MgdG8gYSByZXNvdXJjZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVudGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfVW50YWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIG9uZSBvciBtb3JlIHRhZ3MgZnJvbSBhIHJlc291cmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVXBkYXRlQXVkaXRTdHJlYW1Db25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZUF1ZGl0U3RyZWFtQ29uZmlndXJhdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBhdWRpdCBzdHJlYW0gY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZmxlZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZUNvbXBhbnlOZXR3b3JrQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9VcGRhdGVDb21wYW55TmV0d29ya0NvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgY29tcGFueSBuZXR3b3JrIGNvbmZpZ3VyYXRpb24gZm9yIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVEZXZpY2VQb2xpY3lDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZURldmljZVBvbGljeUNvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgZGV2aWNlIHBvbGljeSBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlRG9tYWluTWV0YWRhdGFcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfVXBkYXRlRG9tYWluTWV0YWRhdGEuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgbWV0YWRhdGEgZm9yIGEgZG9tYWluIGFzc29jaWF0ZWQgd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlRmxlZXRNZXRhZGF0YVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9VcGRhdGVGbGVldE1ldGFkYXRhLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIG1ldGFkYXRhIG9mIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImZsZWV0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVJZGVudGl0eVByb3ZpZGVyQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9VcGRhdGVJZGVudGl0eVByb3ZpZGVyQ29uZmlndXJhdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBpZGVudGl0eSBwcm92aWRlciBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJmbGVldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH07XG4gIHByb3RlY3RlZCByZXNvdXJjZVR5cGVzOiBSZXNvdXJjZVR5cGVzID0ge1xuICAgIFwiZmxlZXRcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZmxlZXRcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvd29ya2xpbmstcmVzb3VyY2VzLmh0bWwjRmxlZXRcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTp3b3JrbGluazo6JHtBY2NvdW50fTpmbGVldC8ke0ZsZWV0TmFtZX1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3dvcmtsaW5rXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b253b3JrbGluay5odG1sKS5cbiAgICpcbiAgICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gICAqL1xuICBjb25zdHJ1Y3RvciAoc2lkPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2lkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgYSBkb21haW4gd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfQXNzb2NpYXRlRG9tYWluLmh0bWxcbiAgICovXG4gIHB1YmxpYyBhc3NvY2lhdGVEb21haW4oKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOkFzc29jaWF0ZURvbWFpbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhIHdlYnNpdGUgYXV0aG9yaXphdGlvbiBwcm92aWRlciB3aXRoIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9Bc3NvY2lhdGVXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyBhc3NvY2lhdGVXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpBc3NvY2lhdGVXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGEgd2Vic2l0ZSBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkgd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfQXNzb2NpYXRlV2Vic2l0ZUNlcnRpZmljYXRlQXV0aG9yaXR5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBhc3NvY2lhdGVXZWJzaXRlQ2VydGlmaWNhdGVBdXRob3JpdHkoKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOkFzc29jaWF0ZVdlYnNpdGVDZXJ0aWZpY2F0ZUF1dGhvcml0eScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlRmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZUZsZWV0KCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpDcmVhdGVGbGVldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGVsZXRlRmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZUZsZWV0KCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpEZWxldGVGbGVldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIHRoZSBhdWRpdCBzdHJlYW0gY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVBdWRpdFN0cmVhbUNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlQXVkaXRTdHJlYW1Db25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpEZXNjcmliZUF1ZGl0U3RyZWFtQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIHRoZSBjb21wYW55IG5ldHdvcmsgY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVDb21wYW55TmV0d29ya0NvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlQ29tcGFueU5ldHdvcmtDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpEZXNjcmliZUNvbXBhbnlOZXR3b3JrQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGRldGFpbHMgb2YgYSBkZXZpY2UgYXNzb2NpYXRlZCB3aXRoIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlRGV2aWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZURldmljZSgpIHtcbiAgICB0aGlzLmFkZCgnd29ya2xpbms6RGVzY3JpYmVEZXZpY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSB0aGUgZGV2aWNlIHBvbGljeSBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9EZXNjcmliZURldmljZVBvbGljeUNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlRGV2aWNlUG9saWN5Q29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnd29ya2xpbms6RGVzY3JpYmVEZXZpY2VQb2xpY3lDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgZGV0YWlscyBhYm91dCBhIGRvbWFpbiBhc3NvY2lhdGVkIHdpdGggYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVEb21haW4uaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlRG9tYWluKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpEZXNjcmliZURvbWFpbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIG1ldGFkYXRhIG9mIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlRmxlZXRNZXRhZGF0YS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVGbGVldE1ldGFkYXRhKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpEZXNjcmliZUZsZWV0TWV0YWRhdGEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSB0aGUgaWRlbnRpdHkgcHJvdmlkZXIgY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVJZGVudGl0eVByb3ZpZGVyQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVJZGVudGl0eVByb3ZpZGVyQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnd29ya2xpbms6RGVzY3JpYmVJZGVudGl0eVByb3ZpZGVyQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGEgd2Vic2l0ZSBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkgYXNzb2NpYXRlZCB3aXRoIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlV2Vic2l0ZUNlcnRpZmljYXRlQXV0aG9yaXR5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZVdlYnNpdGVDZXJ0aWZpY2F0ZUF1dGhvcml0eSgpIHtcbiAgICB0aGlzLmFkZCgnd29ya2xpbms6RGVzY3JpYmVXZWJzaXRlQ2VydGlmaWNhdGVBdXRob3JpdHknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYSBkb21haW4gZnJvbSBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGlzYXNzb2NpYXRlRG9tYWluLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkaXNhc3NvY2lhdGVEb21haW4oKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOkRpc2Fzc29jaWF0ZURvbWFpbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSBhIHdlYnNpdGUgYXV0aG9yaXphdGlvbiBwcm92aWRlciBmcm9tIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9EaXNhc3NvY2lhdGVXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkaXNhc3NvY2lhdGVXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpEaXNhc3NvY2lhdGVXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIGEgd2Vic2l0ZSBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkgZnJvbSBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfRGlzYXNzb2NpYXRlV2Vic2l0ZUNlcnRpZmljYXRlQXV0aG9yaXR5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkaXNhc3NvY2lhdGVXZWJzaXRlQ2VydGlmaWNhdGVBdXRob3JpdHkoKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOkRpc2Fzc29jaWF0ZVdlYnNpdGVDZXJ0aWZpY2F0ZUF1dGhvcml0eScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGRldmljZXMgYXNzb2NpYXRlZCB3aXRoIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0xpc3REZXZpY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0RGV2aWNlcygpIHtcbiAgICB0aGlzLmFkZCgnd29ya2xpbms6TGlzdERldmljZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBhc3NvY2lhdGVkIGRvbWFpbnMgZm9yIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0xpc3REb21haW5zLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0RG9tYWlucygpIHtcbiAgICB0aGlzLmFkZCgnd29ya2xpbms6TGlzdERvbWFpbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBBbWF6b24gV29ya0xpbmsgZmxlZXRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0xpc3RGbGVldHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RGbGVldHMoKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOkxpc3RGbGVldHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3MgZm9yIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgd2Vic2l0ZSBhdXRob3JpemF0aW9uIHByb3ZpZGVycyBmb3IgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfTGlzdFdlYnNpdGVBdXRob3JpemF0aW9uUHJvdmlkZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0V2Vic2l0ZUF1dGhvcml6YXRpb25Qcm92aWRlcnMoKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOkxpc3RXZWJzaXRlQXV0aG9yaXphdGlvblByb3ZpZGVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHdlYnNpdGUgY2VydGlmaWNhdGUgYXV0aG9yaXRpZXMgYXNzb2NpYXRlZCB3aXRoIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX0xpc3RXZWJzaXRlQ2VydGlmaWNhdGVBdXRob3JpdGllcy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFdlYnNpdGVDZXJ0aWZpY2F0ZUF1dGhvcml0aWVzKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpMaXN0V2Vic2l0ZUNlcnRpZmljYXRlQXV0aG9yaXRpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXN0b3JlIGFjY2VzcyB0byBhIGRvbWFpbiBhc3NvY2lhdGVkIHdpdGggYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX1Jlc3RvcmVEb21haW5BY2Nlc3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHJlc3RvcmVEb21haW5BY2Nlc3MoKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOlJlc3RvcmVEb21haW5BY2Nlc3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXZva2UgYWNjZXNzIHRvIGEgZG9tYWluIGFzc29jaWF0ZWQgd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfUmV2b2tlRG9tYWluQWNjZXNzLmh0bWxcbiAgICovXG4gIHB1YmxpYyByZXZva2VEb21haW5BY2Nlc3MoKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOlJldm9rZURvbWFpbkFjY2VzcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNpZ24gb3V0IGEgdXNlciBmcm9tIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9TaWduT3V0VXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgc2lnbk91dFVzZXIoKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOlNpZ25PdXRVc2VyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG9uZSBvciBtb3JlIHRhZ3MgdG8gYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0YWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnd29ya2xpbms6VGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgb25lIG9yIG1vcmUgdGFncyBmcm9tIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1bnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBhdWRpdCBzdHJlYW0gY29uZmlndXJhdGlvbiBmb3IgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZUF1ZGl0U3RyZWFtQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlQXVkaXRTdHJlYW1Db25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpVcGRhdGVBdWRpdFN0cmVhbUNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGNvbXBhbnkgbmV0d29yayBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfVXBkYXRlQ29tcGFueU5ldHdvcmtDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVDb21wYW55TmV0d29ya0NvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOlVwZGF0ZUNvbXBhbnlOZXR3b3JrQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgZGV2aWNlIHBvbGljeSBjb25maWd1cmF0aW9uIGZvciBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfVXBkYXRlRGV2aWNlUG9saWN5Q29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlRGV2aWNlUG9saWN5Q29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnd29ya2xpbms6VXBkYXRlRGV2aWNlUG9saWN5Q29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgbWV0YWRhdGEgZm9yIGEgZG9tYWluIGFzc29jaWF0ZWQgd2l0aCBhbiBBbWF6b24gV29ya0xpbmsgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS9BUElfVXBkYXRlRG9tYWluTWV0YWRhdGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZURvbWFpbk1ldGFkYXRhKCkge1xuICAgIHRoaXMuYWRkKCd3b3JrbGluazpVcGRhdGVEb21haW5NZXRhZGF0YScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgbWV0YWRhdGEgb2YgYW4gQW1hem9uIFdvcmtMaW5rIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dvcmtsaW5rL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZUZsZWV0TWV0YWRhdGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZUZsZWV0TWV0YWRhdGEoKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOlVwZGF0ZUZsZWV0TWV0YWRhdGEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGlkZW50aXR5IHByb3ZpZGVyIGNvbmZpZ3VyYXRpb24gZm9yIGFuIEFtYXpvbiBXb3JrTGluayBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93b3JrbGluay9sYXRlc3QvYXBpL0FQSV9VcGRhdGVJZGVudGl0eVByb3ZpZGVyQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlSWRlbnRpdHlQcm92aWRlckNvbmZpZ3VyYXRpb24oKSB7XG4gICAgdGhpcy5hZGQoJ3dvcmtsaW5rOlVwZGF0ZUlkZW50aXR5UHJvdmlkZXJDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZmxlZXQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd29ya2xpbmsvbGF0ZXN0L2FwaS93b3JrbGluay1yZXNvdXJjZXMuaHRtbCNGbGVldFxuICAgKlxuICAgKiBAcGFyYW0gZmxlZXROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGZsZWV0TmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkZsZWV0KGZsZWV0TmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06d29ya2xpbms6OiR7QWNjb3VudH06ZmxlZXQvJHtGbGVldE5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtGbGVldE5hbWV9JywgZmxlZXROYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWN0aW9ucyBiYXNlZCBvbiB0aGUgcHJlc2VuY2Ugb2YgdGFnIGtleS12YWx1ZSBwYWlycyBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVxdWVzdHRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLmNyZWF0ZUZsZWV0KClcbiAgICogLSAuZGVsZXRlRmxlZXQoKVxuICAgKiAtIC5kZXNjcmliZUZsZWV0TWV0YWRhdGEoKVxuICAgKiAtIC50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRhZyBrZXktdmFsdWUgcGFpcnMgYXR0YWNoZWQgdG8gdGhlIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGZsZWV0XG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGhlIHByZXNlbmNlIG9mIHRhZyBrZXlzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy10YWdrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAuY3JlYXRlRmxlZXQoKVxuICAgKiAtIC5kZWxldGVGbGVldCgpXG4gICAqIC0gLmRlc2NyaWJlRmxlZXRNZXRhZGF0YSgpXG4gICAqIC0gLnRhZ1Jlc291cmNlKClcbiAgICogLSAudW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cbn1cbiJdfQ==