"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Artifact = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [artifact](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsartifact.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Artifact extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [artifact](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsartifact.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'artifact';
        this.actionList = {
            "AcceptAgreement": {
                "url": "https://docs.aws.amazon.com/artifact/latest/ug/managingagreements.html",
                "description": "Grants permission to accept an AWS agreement that has not yet been accepted by the customer account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "agreement": {
                        "required": true
                    }
                }
            },
            "DownloadAgreement": {
                "url": "https://docs.aws.amazon.com/artifact/latest/ug/managingagreements.html",
                "description": "Grants permission to download an AWS agreement that has not yet been accepted or a customer agreement that has been accepted by the customer account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "agreement": {
                        "required": false
                    },
                    "customer-agreement": {
                        "required": false
                    }
                }
            },
            "Get": {
                "url": "https://docs.aws.amazon.com/artifact/latest/ug/getting-started.html",
                "description": "Grants permission to download an AWS compliance report package.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "report-package": {
                        "required": true
                    }
                }
            },
            "TerminateAgreement": {
                "url": "https://docs.aws.amazon.com/artifact/latest/ug/managingagreements.html",
                "description": "Grants permission to terminate a customer agreement that was previously accepted by the customer account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "customer-agreement": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "report-package": {
                "name": "report-package",
                "url": "https://docs.aws.amazon.com/artifact/latest/ug/what-is-aws-artifact.html",
                "arn": "arn:${Partition}:artifact:::report-package/${ResourceName}",
                "conditionKeys": []
            },
            "customer-agreement": {
                "name": "customer-agreement",
                "url": "https://docs.aws.amazon.com/artifact/latest/ug/$managingagreements.html",
                "arn": "arn:${Partition}:artifact::${Account}:customer-agreement/${ResourceName}",
                "conditionKeys": []
            },
            "agreement": {
                "name": "agreement",
                "url": "https://docs.aws.amazon.com/artifact/latest/ug/managingagreements.html",
                "arn": "arn:${Partition}:artifact:::agreement/${ResourceName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept an AWS agreement that has not yet been accepted by the customer account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managingagreements.html
     */
    acceptAgreement() {
        this.add('artifact:AcceptAgreement');
        return this;
    }
    /**
     * Grants permission to download an AWS agreement that has not yet been accepted or a customer agreement that has been accepted by the customer account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managingagreements.html
     */
    downloadAgreement() {
        this.add('artifact:DownloadAgreement');
        return this;
    }
    /**
     * Grants permission to download an AWS compliance report package.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/getting-started.html
     */
    get() {
        this.add('artifact:Get');
        return this;
    }
    /**
     * Grants permission to terminate a customer agreement that was previously accepted by the customer account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managingagreements.html
     */
    terminateAgreement() {
        this.add('artifact:TerminateAgreement');
        return this;
    }
    /**
     * Adds a resource of type report-package to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/what-is-aws-artifact.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReportPackage(resourceName, partition) {
        var arn = 'arn:${Partition}:artifact:::report-package/${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type customer-agreement to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/$managingagreements.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCustomerAgreement(resourceName, account, partition) {
        var arn = 'arn:${Partition}:artifact::${Account}:customer-agreement/${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type agreement to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managingagreements.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAgreement(resourceName, partition) {
        var arn = 'arn:${Partition}:artifact:::agreement/${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Artifact = Artifact;
//# sourceMappingURL=data:application/json;base64,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