"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediaconnect = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediaconnect](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediaconnect extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediaconnect](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconnect.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediaconnect';
        this.actionList = {
            "AddFlowOutputs": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs.html",
                "description": "Grants permission to add outputs to any flow.",
                "accessLevel": "Write"
            },
            "CreateFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html",
                "description": "Grants permission to create flows.",
                "accessLevel": "Write"
            },
            "DeleteFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html",
                "description": "Grants permission to delete flows.",
                "accessLevel": "Write"
            },
            "DescribeFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html",
                "description": "Grants permission to display the details of a flow including the flow ARN, name, and Availability Zone, as well as details about the source, outputs, and entitlements.",
                "accessLevel": "Read"
            },
            "GrantFlowEntitlements": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements.html",
                "description": "Grants permission to grant entitlements on any flow.",
                "accessLevel": "Write"
            },
            "ListEntitlements": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-entitlements.html",
                "description": "Grants permission to display a list of all entitlements that have been granted to the account.",
                "accessLevel": "List"
            },
            "ListFlows": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html",
                "description": "Grants permission to display a list of flows that are associated with this account.",
                "accessLevel": "List"
            },
            "RemoveFlowOutput": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html",
                "description": "Grants permission to remove outputs from any flow.",
                "accessLevel": "Write"
            },
            "RevokeFlowEntitlement": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html",
                "description": "Grants permission to revoke entitlements on any flow.",
                "accessLevel": "Write"
            },
            "StartFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-start-flowarn.html",
                "description": "Grants permission to start flows.",
                "accessLevel": "Write"
            },
            "StopFlow": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-stop-flowarn.html",
                "description": "Grants permission to stop flows.",
                "accessLevel": "Write"
            },
            "UpdateFlowEntitlement": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html",
                "description": "Grants permission to update entitlements on any flow.",
                "accessLevel": "Write"
            },
            "UpdateFlowOutput": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html",
                "description": "Grants permission to update outputs on any flow.",
                "accessLevel": "Write"
            },
            "UpdateFlowSource": {
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source-sourcearn.html",
                "description": "Grants permission to update the source of any flow.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Entitlement": {
                "name": "Entitlement",
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/ug/entitlements.html",
                "arn": "arn:${Partition}:mediaconnect:${Region}:${Account}:entitlement:${FlowId}:${EntitlementName}",
                "conditionKeys": []
            },
            "Flow": {
                "name": "Flow",
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/ug/flows.html",
                "arn": "arn:${Partition}:mediaconnect:${Region}:${Account}:flow:${FlowId}:${FlowName}",
                "conditionKeys": []
            },
            "Output": {
                "name": "Output",
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/ug/outputs.html",
                "arn": "arn:${Partition}:mediaconnect:${Region}:${Account}:output:${OutputId}:${OutputName}",
                "conditionKeys": []
            },
            "Source": {
                "name": "Source",
                "url": "https://docs.aws.amazon.com/mediaconnect/latest/ug/sources.html",
                "arn": "arn:${Partition}:mediaconnect:${Region}:${Account}:source:${SourceId}:${SourceName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add outputs to any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs.html
     */
    addFlowOutputs() {
        this.add('mediaconnect:AddFlowOutputs');
        return this;
    }
    /**
     * Grants permission to create flows.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    createFlow() {
        this.add('mediaconnect:CreateFlow');
        return this;
    }
    /**
     * Grants permission to delete flows.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html
     */
    deleteFlow() {
        this.add('mediaconnect:DeleteFlow');
        return this;
    }
    /**
     * Grants permission to display the details of a flow including the flow ARN, name, and Availability Zone, as well as details about the source, outputs, and entitlements.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html
     */
    describeFlow() {
        this.add('mediaconnect:DescribeFlow');
        return this;
    }
    /**
     * Grants permission to grant entitlements on any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements.html
     */
    grantFlowEntitlements() {
        this.add('mediaconnect:GrantFlowEntitlements');
        return this;
    }
    /**
     * Grants permission to display a list of all entitlements that have been granted to the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-entitlements.html
     */
    listEntitlements() {
        this.add('mediaconnect:ListEntitlements');
        return this;
    }
    /**
     * Grants permission to display a list of flows that are associated with this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    listFlows() {
        this.add('mediaconnect:ListFlows');
        return this;
    }
    /**
     * Grants permission to remove outputs from any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html
     */
    removeFlowOutput() {
        this.add('mediaconnect:RemoveFlowOutput');
        return this;
    }
    /**
     * Grants permission to revoke entitlements on any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html
     */
    revokeFlowEntitlement() {
        this.add('mediaconnect:RevokeFlowEntitlement');
        return this;
    }
    /**
     * Grants permission to start flows.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-start-flowarn.html
     */
    startFlow() {
        this.add('mediaconnect:StartFlow');
        return this;
    }
    /**
     * Grants permission to stop flows.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-stop-flowarn.html
     */
    stopFlow() {
        this.add('mediaconnect:StopFlow');
        return this;
    }
    /**
     * Grants permission to update entitlements on any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html
     */
    updateFlowEntitlement() {
        this.add('mediaconnect:UpdateFlowEntitlement');
        return this;
    }
    /**
     * Grants permission to update outputs on any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html
     */
    updateFlowOutput() {
        this.add('mediaconnect:UpdateFlowOutput');
        return this;
    }
    /**
     * Grants permission to update the source of any flow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source-sourcearn.html
     */
    updateFlowSource() {
        this.add('mediaconnect:UpdateFlowSource');
        return this;
    }
    /**
     * Adds a resource of type Entitlement to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/entitlements.html
     *
     * @param flowId - Identifier for the flowId.
     * @param entitlementName - Identifier for the entitlementName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEntitlement(flowId, entitlementName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconnect:${Region}:${Account}:entitlement:${FlowId}:${EntitlementName}';
        arn = arn.replace('${FlowId}', flowId);
        arn = arn.replace('${EntitlementName}', entitlementName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Flow to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/flows.html
     *
     * @param flowId - Identifier for the flowId.
     * @param flowName - Identifier for the flowName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFlow(flowId, flowName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconnect:${Region}:${Account}:flow:${FlowId}:${FlowName}';
        arn = arn.replace('${FlowId}', flowId);
        arn = arn.replace('${FlowName}', flowName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Output to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/outputs.html
     *
     * @param outputId - Identifier for the outputId.
     * @param outputName - Identifier for the outputName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOutput(outputId, outputName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconnect:${Region}:${Account}:output:${OutputId}:${OutputName}';
        arn = arn.replace('${OutputId}', outputId);
        arn = arn.replace('${OutputName}', outputName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Source to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/sources.html
     *
     * @param sourceId - Identifier for the sourceId.
     * @param sourceName - Identifier for the sourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSource(sourceId, sourceName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconnect:${Region}:${Account}:source:${SourceId}:${SourceName}';
        arn = arn.replace('${SourceId}', sourceId);
        arn = arn.replace('${SourceName}', sourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediaconnect = Mediaconnect;
//# sourceMappingURL=data:application/json;base64,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