"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediaconvert = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconvert.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediaconvert extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconvert.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediaconvert';
        this.actionList = {
            "AssociateCertificate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html",
                "description": "Grants permission to associate an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert.",
                "accessLevel": "Write"
            },
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html",
                "description": "Grants permission to cancel an AWS Elemental MediaConvert job that is waiting in queue",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Job": {
                        "required": true
                    }
                }
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html",
                "description": "Grants permission to create and submit an AWS Elemental MediaConvert job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": false
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                }
            },
            "CreateJobTemplate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html",
                "description": "Grants permission to create an AWS Elemental MediaConvert custom job template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePreset": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html",
                "description": "Grants permission to create an AWS Elemental MediaConvert custom output preset",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateQueue": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html",
                "description": "Grants permission to create an AWS Elemental MediaConvert job queue",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteJobTemplate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html",
                "description": "Grants permission to delete an AWS Elemental MediaConvert custom job template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": true
                    }
                }
            },
            "DeletePreset": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html",
                "description": "Grants permission to delete an AWS Elemental MediaConvert custom output preset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Preset": {
                        "required": true
                    }
                }
            },
            "DeleteQueue": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html",
                "description": "Grants permission to delete an AWS Elemental MediaConvert job queue",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Queue": {
                        "required": true
                    }
                }
            },
            "DescribeEndpoints": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/endpoints.html",
                "description": "Grants permission to subscribe to the AWS Elemental MediaConvert service, by sending a request for an account-specific endpoint. All transcoding requests must be sent to the endpoint that the service returns.",
                "accessLevel": "List"
            },
            "DisassociateCertificate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates-arn.html",
                "description": "Grants permission to remove an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and an AWS Elemental MediaConvert resource.",
                "accessLevel": "Write"
            },
            "GetJob": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html",
                "description": "Grants permission to get an AWS Elemental MediaConvert job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Job": {
                        "required": true
                    }
                }
            },
            "GetJobTemplate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html",
                "description": "Grants permission to get an AWS Elemental MediaConvert job template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": true
                    }
                }
            },
            "GetPreset": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html",
                "description": "Grants permission to get an AWS Elemental MediaConvert output preset",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Preset": {
                        "required": true
                    }
                }
            },
            "GetQueue": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html",
                "description": "Grants permission to get an AWS Elemental MediaConvert job queue",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Queue": {
                        "required": true
                    }
                }
            },
            "ListJobTemplates": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html",
                "description": "Grants permission to list AWS Elemental MediaConvert job templates",
                "accessLevel": "List"
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html",
                "description": "Grants permission to list AWS Elemental MediaConvert jobs",
                "accessLevel": "List",
                "resourceTypes": {
                    "Queue": {
                        "required": false
                    }
                }
            },
            "ListPresets": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html",
                "description": "Grants permission to list AWS Elemental MediaConvert output presets",
                "accessLevel": "List"
            },
            "ListQueues": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html",
                "description": "Grants permission to list AWS Elemental MediaConvert job queues",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html",
                "description": "Grants permission to retrieve the tags for a MediaConvert queue, preset, or job template",
                "accessLevel": "Read",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": false
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html",
                "description": "Grants permission to add tags to a MediaConvert queue, preset, or job template",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": false
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html",
                "description": "Grants permission to remove tags from a MediaConvert queue, preset, or job template",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": false
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateJobTemplate": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html",
                "description": "Grants permission to update an AWS Elemental MediaConvert custom job template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "JobTemplate": {
                        "required": true
                    },
                    "Preset": {
                        "required": false
                    },
                    "Queue": {
                        "required": false
                    }
                }
            },
            "UpdatePreset": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html",
                "description": "Grants permission to update an AWS Elemental MediaConvert custom output preset",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Preset": {
                        "required": true
                    }
                }
            },
            "UpdateQueue": {
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html",
                "description": "Grants permission to update an AWS Elemental MediaConvert job queue",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Queue": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "Job": {
                "name": "Job",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:jobs/${JobId}",
                "conditionKeys": []
            },
            "Queue": {
                "name": "Queue",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:queues/${QueueName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Preset": {
                "name": "Preset",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:presets/${PresetName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "JobTemplate": {
                "name": "JobTemplate",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:jobTemplates/${JobTemplateName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CertificateAssociation": {
                "name": "CertificateAssociation",
                "url": "https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html",
                "arn": "arn:${Partition}:mediaconvert:${Region}:${Account}:certificates/${CertificateArn}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     */
    associateCertificate() {
        this.add('mediaconvert:AssociateCertificate');
        return this;
    }
    /**
     * Grants permission to cancel an AWS Elemental MediaConvert job that is waiting in queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    cancelJob() {
        this.add('mediaconvert:CancelJob');
        return this;
    }
    /**
     * Grants permission to create and submit an AWS Elemental MediaConvert job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    createJob() {
        this.add('mediaconvert:CreateJob');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    createJobTemplate() {
        this.add('mediaconvert:CreateJobTemplate');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    createPreset() {
        this.add('mediaconvert:CreatePreset');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    createQueue() {
        this.add('mediaconvert:CreateQueue');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    deleteJobTemplate() {
        this.add('mediaconvert:DeleteJobTemplate');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    deletePreset() {
        this.add('mediaconvert:DeletePreset');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    deleteQueue() {
        this.add('mediaconvert:DeleteQueue');
        return this;
    }
    /**
     * Grants permission to subscribe to the AWS Elemental MediaConvert service, by sending a request for an account-specific endpoint. All transcoding requests must be sent to the endpoint that the service returns.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/endpoints.html
     */
    describeEndpoints() {
        this.add('mediaconvert:DescribeEndpoints');
        return this;
    }
    /**
     * Grants permission to remove an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and an AWS Elemental MediaConvert resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates-arn.html
     */
    disassociateCertificate() {
        this.add('mediaconvert:DisassociateCertificate');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    getJob() {
        this.add('mediaconvert:GetJob');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    getJobTemplate() {
        this.add('mediaconvert:GetJobTemplate');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert output preset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    getPreset() {
        this.add('mediaconvert:GetPreset');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job queue
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    getQueue() {
        this.add('mediaconvert:GetQueue');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     */
    listJobTemplates() {
        this.add('mediaconvert:ListJobTemplates');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    listJobs() {
        this.add('mediaconvert:ListJobs');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert output presets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    listPresets() {
        this.add('mediaconvert:ListPresets');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job queues
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    listQueues() {
        this.add('mediaconvert:ListQueues');
        return this;
    }
    /**
     * Grants permission to retrieve the tags for a MediaConvert queue, preset, or job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    listTagsForResource() {
        this.add('mediaconvert:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to add tags to a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     */
    tagResource() {
        this.add('mediaconvert:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    untagResource() {
        this.add('mediaconvert:UntagResource');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    updateJobTemplate() {
        this.add('mediaconvert:UpdateJobTemplate');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    updatePreset() {
        this.add('mediaconvert:UpdatePreset');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    updateQueue() {
        this.add('mediaconvert:UpdateQueue');
        return this;
    }
    /**
     * Adds a resource of type Job to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:jobs/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Queue to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     *
     * @param queueName - Identifier for the queueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQueue(queueName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:queues/${QueueName}';
        arn = arn.replace('${QueueName}', queueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Preset to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     *
     * @param presetName - Identifier for the presetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPreset(presetName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:presets/${PresetName}';
        arn = arn.replace('${PresetName}', presetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type JobTemplate to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     *
     * @param jobTemplateName - Identifier for the jobTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJobTemplate(jobTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:jobTemplates/${JobTemplateName}';
        arn = arn.replace('${JobTemplateName}', jobTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CertificateAssociation to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     *
     * @param certificateArn - Identifier for the certificateArn.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCertificateAssociation(certificateArn, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:certificates/${CertificateArn}';
        arn = arn.replace('${CertificateArn}', certificateArn);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .createJobTemplate()
     * - .createPreset()
     * - .createQueue()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to resource types:
     * - Queue
     * - Preset
     * - JobTemplate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .createJobTemplate()
     * - .createPreset()
     * - .createQueue()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Mediaconvert = Mediaconvert;
//# sourceMappingURL=data:application/json;base64,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