"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediapackageVod = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediapackage-vod](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediapackagevod.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class MediapackageVod extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediapackage-vod](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediapackagevod.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediapackage-vod';
        this.actionList = {
            "CreateAsset": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetspost",
                "description": "Grants permission to create an asset in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePackagingConfiguration": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationspost",
                "description": "Grants permission to create a packaging configuration in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePackagingGroup": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupspost",
                "description": "Grants permission to create a packaging group in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAsset": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-iddelete",
                "description": "Grants permission to delete an asset in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assets": {
                        "required": true
                    }
                }
            },
            "DeletePackagingConfiguration": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-iddelete",
                "description": "Grants permission to delete a packaging configuration in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "resourceTypes": {
                    "packaging-configurations": {
                        "required": true
                    }
                }
            },
            "DeletePackagingGroup": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-iddelete",
                "description": "Grants permission to delete a packaging group in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "resourceTypes": {
                    "packaging-groups": {
                        "required": true
                    }
                }
            },
            "DescribeAsset": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-idget",
                "description": "Grants permission to view the details of an asset in AWS Elemental MediaPackage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "assets": {
                        "required": true
                    }
                }
            },
            "DescribePackagingConfiguration": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-idget",
                "description": "Grants permission to view the details of a packaging configuration in AWS Elemental MediaPackage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "packaging-configurations": {
                        "required": true
                    }
                }
            },
            "DescribePackagingGroup": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-idget",
                "description": "Grants permission to view the details of a packaging group in AWS Elemental MediaPackage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "packaging-groups": {
                        "required": true
                    }
                }
            },
            "ListAssets": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetsget",
                "description": "Grants permission to view a list of assets in AWS Elemental MediaPackage",
                "accessLevel": "List"
            },
            "ListPackagingConfigurations": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationsget",
                "description": "Grants permission to view a list of packaging configurations in AWS Elemental MediaPackage",
                "accessLevel": "List"
            },
            "ListPackagingGroups": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupsget",
                "description": "Grants permission to view a list of packaging groups in AWS Elemental MediaPackage",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnget",
                "description": "Grants permission to list the tags assigned to a PackagingGroup, PackagingConfiguration, or Asset.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "assets": {
                        "required": false
                    },
                    "packaging-configurations": {
                        "required": false
                    },
                    "packaging-groups": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnpost",
                "description": "Grants permission to assign tags to a PackagingGroup, PackagingConfiguration, or Asset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assets": {
                        "required": false
                    },
                    "packaging-configurations": {
                        "required": false
                    },
                    "packaging-groups": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arndelete",
                "description": "Grants permission to delete tags from a PackagingGroup, PackagingConfiguration, or Asset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assets": {
                        "required": false
                    },
                    "packaging-configurations": {
                        "required": false
                    },
                    "packaging-groups": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "assets": {
                "name": "assets",
                "url": "https://docs.aws.amazon.com/mediapackage/latest/ug/asset.html",
                "arn": "arn:${Partition}:mediapackage-vod:${Region}:${Account}:assets/${AssetIdentifier}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "packaging-configurations": {
                "name": "packaging-configurations",
                "url": "https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-cfig.html",
                "arn": "arn:${Partition}:mediapackage-vod:${Region}:${Account}:packaging-configurations/${PackagingConfigurationIdentifier}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "packaging-groups": {
                "name": "packaging-groups",
                "url": "https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-group.html",
                "arn": "arn:${Partition}:mediapackage-vod:${Region}:${Account}:packaging-groups/${PackagingGroupIdentifier}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create an asset in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetspost
     */
    createAsset() {
        this.add('mediapackage-vod:CreateAsset');
        return this;
    }
    /**
     * Grants permission to create a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationspost
     */
    createPackagingConfiguration() {
        this.add('mediapackage-vod:CreatePackagingConfiguration');
        return this;
    }
    /**
     * Grants permission to create a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupspost
     */
    createPackagingGroup() {
        this.add('mediapackage-vod:CreatePackagingGroup');
        return this;
    }
    /**
     * Grants permission to delete an asset in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-iddelete
     */
    deleteAsset() {
        this.add('mediapackage-vod:DeleteAsset');
        return this;
    }
    /**
     * Grants permission to delete a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-iddelete
     */
    deletePackagingConfiguration() {
        this.add('mediapackage-vod:DeletePackagingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-iddelete
     */
    deletePackagingGroup() {
        this.add('mediapackage-vod:DeletePackagingGroup');
        return this;
    }
    /**
     * Grants permission to view the details of an asset in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-idget
     */
    describeAsset() {
        this.add('mediapackage-vod:DescribeAsset');
        return this;
    }
    /**
     * Grants permission to view the details of a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-idget
     */
    describePackagingConfiguration() {
        this.add('mediapackage-vod:DescribePackagingConfiguration');
        return this;
    }
    /**
     * Grants permission to view the details of a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-idget
     */
    describePackagingGroup() {
        this.add('mediapackage-vod:DescribePackagingGroup');
        return this;
    }
    /**
     * Grants permission to view a list of assets in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetsget
     */
    listAssets() {
        this.add('mediapackage-vod:ListAssets');
        return this;
    }
    /**
     * Grants permission to view a list of packaging configurations in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationsget
     */
    listPackagingConfigurations() {
        this.add('mediapackage-vod:ListPackagingConfigurations');
        return this;
    }
    /**
     * Grants permission to view a list of packaging groups in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupsget
     */
    listPackagingGroups() {
        this.add('mediapackage-vod:ListPackagingGroups');
        return this;
    }
    /**
     * Grants permission to list the tags assigned to a PackagingGroup, PackagingConfiguration, or Asset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnget
     */
    listTagsForResource() {
        this.add('mediapackage-vod:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to assign tags to a PackagingGroup, PackagingConfiguration, or Asset.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnpost
     */
    tagResource() {
        this.add('mediapackage-vod:TagResource');
        return this;
    }
    /**
     * Grants permission to delete tags from a PackagingGroup, PackagingConfiguration, or Asset.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arndelete
     */
    untagResource() {
        this.add('mediapackage-vod:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type assets to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/asset.html
     *
     * @param assetIdentifier - Identifier for the assetIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssets(assetIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage-vod:${Region}:${Account}:assets/${AssetIdentifier}';
        arn = arn.replace('${AssetIdentifier}', assetIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type packaging-configurations to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-cfig.html
     *
     * @param packagingConfigurationIdentifier - Identifier for the packagingConfigurationIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPackagingConfigurations(packagingConfigurationIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage-vod:${Region}:${Account}:packaging-configurations/${PackagingConfigurationIdentifier}';
        arn = arn.replace('${PackagingConfigurationIdentifier}', packagingConfigurationIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type packaging-groups to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-group.html
     *
     * @param packagingGroupIdentifier - Identifier for the packagingGroupIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPackagingGroups(packagingGroupIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage-vod:${Region}:${Account}:packaging-groups/${PackagingGroupIdentifier}';
        arn = arn.replace('${PackagingGroupIdentifier}', packagingGroupIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createAsset()
     * - .createPackagingConfiguration()
     * - .createPackagingGroup()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - assets
     * - packaging-configurations
     * - packaging-groups
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createAsset()
     * - .createPackagingConfiguration()
     * - .createPackagingGroup()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.MediapackageVod = MediapackageVod;
//# sourceMappingURL=data:application/json;base64,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