"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Health = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [health](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awshealthapisandnotifications.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Health extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [health](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awshealthapisandnotifications.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'health';
        this.actionList = {
            "DescribeAffectedAccountsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html",
                "description": "Gets a list of accounts that have been affected by the specified events in organization.",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DescribeAffectedEntities": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html",
                "description": "Gets a list of entities that have been affected by the specified events.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event": {
                        "required": true
                    }
                },
                "conditions": [
                    "health:eventTypeCode",
                    "health:service"
                ]
            },
            "DescribeAffectedEntitiesForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html",
                "description": "Gets a list of entities that have been affected by the specified events and accounts in organization.",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DescribeEntityAggregates": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html",
                "description": "Returns the number of entities that are affected by each of the specified events.",
                "accessLevel": "Read"
            },
            "DescribeEventAggregates": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html",
                "description": "Returns the number of events of each event type (issue, scheduled change, and account notification).",
                "accessLevel": "Read"
            },
            "DescribeEventDetails": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html",
                "description": "Returns detailed information about one or more specified events.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event": {
                        "required": true
                    }
                },
                "conditions": [
                    "health:eventTypeCode",
                    "health:service"
                ]
            },
            "DescribeEventDetailsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html",
                "description": "Returns detailed information about one or more specified events for provided accounts in organization.",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DescribeEventTypes": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html",
                "description": "Returns the event types that meet the specified filter criteria.",
                "accessLevel": "Read"
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html",
                "description": "Returns information about events that meet the specified filter criteria.",
                "accessLevel": "Read"
            },
            "DescribeEventsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html",
                "description": "Returns information about events that meet the specified filter criteria in organization.",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DescribeHealthServiceStatusForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html",
                "description": "Returns the status of enabling or disabling the Organizational View feature",
                "accessLevel": "Permissions management",
                "dependentActions": [
                    "organizations:ListAccounts"
                ]
            },
            "DisableHealthServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html",
                "description": "Disables the Organizational View feature.",
                "accessLevel": "Permissions management",
                "dependentActions": [
                    "organizations:DisableAWSServiceAccess",
                    "organizations:ListAccounts"
                ]
            },
            "EnableHealthServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html",
                "description": "Enables the Organizational View feature.",
                "accessLevel": "Permissions management",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole",
                    "organizations:EnableAWSServiceAccess",
                    "organizations:ListAccounts"
                ]
            }
        };
        this.resourceTypes = {
            "event": {
                "name": "event",
                "url": "https://docs.aws.amazon.com/health/latest/ug/supported-operations.html",
                "arn": "arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Gets a list of accounts that have been affected by the specified events in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html
     */
    describeAffectedAccountsForOrganization() {
        this.add('health:DescribeAffectedAccountsForOrganization');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEventTypeCode()
     * - .ifService()
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html
     */
    describeAffectedEntities() {
        this.add('health:DescribeAffectedEntities');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events and accounts in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html
     */
    describeAffectedEntitiesForOrganization() {
        this.add('health:DescribeAffectedEntitiesForOrganization');
        return this;
    }
    /**
     * Returns the number of entities that are affected by each of the specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html
     */
    describeEntityAggregates() {
        this.add('health:DescribeEntityAggregates');
        return this;
    }
    /**
     * Returns the number of events of each event type (issue, scheduled change, and account notification).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html
     */
    describeEventAggregates() {
        this.add('health:DescribeEventAggregates');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEventTypeCode()
     * - .ifService()
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html
     */
    describeEventDetails() {
        this.add('health:DescribeEventDetails');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events for provided accounts in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html
     */
    describeEventDetailsForOrganization() {
        this.add('health:DescribeEventDetailsForOrganization');
        return this;
    }
    /**
     * Returns the event types that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html
     */
    describeEventTypes() {
        this.add('health:DescribeEventTypes');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html
     */
    describeEvents() {
        this.add('health:DescribeEvents');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria in organization.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html
     */
    describeEventsForOrganization() {
        this.add('health:DescribeEventsForOrganization');
        return this;
    }
    /**
     * Returns the status of enabling or disabling the Organizational View feature
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html
     */
    describeHealthServiceStatusForOrganization() {
        this.add('health:DescribeHealthServiceStatusForOrganization');
        return this;
    }
    /**
     * Disables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - organizations:DisableAWSServiceAccess
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html
     */
    disableHealthServiceAccessForOrganization() {
        this.add('health:DisableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Enables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListAccounts
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html
     */
    enableHealthServiceAccessForOrganization() {
        this.add('health:EnableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Adds a resource of type event to the statement
     *
     * https://docs.aws.amazon.com/health/latest/ug/supported-operations.html
     *
     * @param service - Identifier for the service.
     * @param eventTypeCode - Identifier for the eventTypeCode.
     * @param eventTypePlusId - Identifier for the eventTypePlusId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvent(service, eventTypeCode, eventTypePlusId, account, region, partition) {
        var arn = 'arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}';
        arn = arn.replace('${Service}', service);
        arn = arn.replace('${EventTypeCode}', eventTypeCode);
        arn = arn.replace('${EventTypePlusId}', eventTypePlusId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The type of event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * Applies to actions:
     * - .describeAffectedEntities()
     * - .describeEventDetails()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEventTypeCode(value, operator) {
        return this.if(`health:eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * The service of the event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * Applies to actions:
     * - .describeAffectedEntities()
     * - .describeEventDetails()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`health:service`, value, operator || 'StringLike');
    }
}
exports.Health = Health;
//# sourceMappingURL=data:application/json;base64,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