"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotevents = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iotevents](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotevents.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotevents extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iotevents](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotevents.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iotevents';
        this.actionList = {
            "BatchPutMessage": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchPutMessage.html",
                "description": "Sends a set of messages to the AWS IoT Events system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "BatchUpdateDetector": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchUpdateDetector.html",
                "description": "Update an detector within the AWS IoT Events system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "CreateDetectorModel": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateDetectorModel.html",
                "description": "Creates a detector model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInput": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateInput.html",
                "description": "Creates an input.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDetectorModel": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteDetectorModel.html",
                "description": "Deletes a detector model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "DeleteInput": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteInput.html",
                "description": "Deletes an input.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "DescribeDetector": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_DescribeDetector.html",
                "description": "Returns information about the specified detector (instance).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "DescribeDetectorModel": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeDetectorModel.html",
                "description": "Describes a detector model.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "DescribeInput": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeInput.html",
                "description": "Describes an input.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "DescribeLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeLoggingOptions.html",
                "description": "Retrieves the current settings of the AWS IoT Events logging options.",
                "accessLevel": "Read"
            },
            "ListDetectorModelVersions": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModelVersions.html",
                "description": "Lists all the versions of a detector model. Only the metadata associated with each detector model version is returned.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "ListDetectorModels": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModels.html",
                "description": "Lists the detector models you have created. Only the metadata associated with each detector model is returned.",
                "accessLevel": "List"
            },
            "ListDetectors": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_ListDetectors.html",
                "description": "Lists detectors (the instances of a detector model).",
                "accessLevel": "List",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "ListInputs": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListInputs.html",
                "description": "Lists the inputs you have created.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListTagsForResource.html",
                "description": "Lists the tags (metadata) which you have assigned to the resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detectorModel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    }
                }
            },
            "PutLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_PutLoggingOptions.html",
                "description": "Sets or updates the AWS IoT Events logging options.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_TagResource.html",
                "description": "Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "detectorModel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_UntagResource.html",
                "description": "Removes the given tags (metadata) from the resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "detectorModel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDetectorModel": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateDetectorModel.html",
                "description": "Updates a detector model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "UpdateInput": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInput.html",
                "description": "Updates an input.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "UpdateInputRouting": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInputRouting.html",
                "description": "Updates input routing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "detectorModel": {
                "name": "detectorModel",
                "url": "https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html",
                "arn": "arn:${Partition}:iotevents:${Region}:${Account}:detectorModel/${DetectorModelName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "input": {
                "name": "input",
                "url": "https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html",
                "arn": "arn:${Partition}:iotevents:${Region}:${Account}:input/${InputName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Sends a set of messages to the AWS IoT Events system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchPutMessage.html
     */
    batchPutMessage() {
        this.add('iotevents:BatchPutMessage');
        return this;
    }
    /**
     * Update an detector within the AWS IoT Events system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchUpdateDetector.html
     */
    batchUpdateDetector() {
        this.add('iotevents:BatchUpdateDetector');
        return this;
    }
    /**
     * Creates a detector model.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateDetectorModel.html
     */
    createDetectorModel() {
        this.add('iotevents:CreateDetectorModel');
        return this;
    }
    /**
     * Creates an input.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateInput.html
     */
    createInput() {
        this.add('iotevents:CreateInput');
        return this;
    }
    /**
     * Deletes a detector model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteDetectorModel.html
     */
    deleteDetectorModel() {
        this.add('iotevents:DeleteDetectorModel');
        return this;
    }
    /**
     * Deletes an input.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteInput.html
     */
    deleteInput() {
        this.add('iotevents:DeleteInput');
        return this;
    }
    /**
     * Returns information about the specified detector (instance).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_DescribeDetector.html
     */
    describeDetector() {
        this.add('iotevents:DescribeDetector');
        return this;
    }
    /**
     * Describes a detector model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeDetectorModel.html
     */
    describeDetectorModel() {
        this.add('iotevents:DescribeDetectorModel');
        return this;
    }
    /**
     * Describes an input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeInput.html
     */
    describeInput() {
        this.add('iotevents:DescribeInput');
        return this;
    }
    /**
     * Retrieves the current settings of the AWS IoT Events logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeLoggingOptions.html
     */
    describeLoggingOptions() {
        this.add('iotevents:DescribeLoggingOptions');
        return this;
    }
    /**
     * Lists all the versions of a detector model. Only the metadata associated with each detector model version is returned.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModelVersions.html
     */
    listDetectorModelVersions() {
        this.add('iotevents:ListDetectorModelVersions');
        return this;
    }
    /**
     * Lists the detector models you have created. Only the metadata associated with each detector model is returned.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModels.html
     */
    listDetectorModels() {
        this.add('iotevents:ListDetectorModels');
        return this;
    }
    /**
     * Lists detectors (the instances of a detector model).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_ListDetectors.html
     */
    listDetectors() {
        this.add('iotevents:ListDetectors');
        return this;
    }
    /**
     * Lists the inputs you have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListInputs.html
     */
    listInputs() {
        this.add('iotevents:ListInputs');
        return this;
    }
    /**
     * Lists the tags (metadata) which you have assigned to the resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('iotevents:ListTagsForResource');
        return this;
    }
    /**
     * Sets or updates the AWS IoT Events logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_PutLoggingOptions.html
     */
    putLoggingOptions() {
        this.add('iotevents:PutLoggingOptions');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('iotevents:TagResource');
        return this;
    }
    /**
     * Removes the given tags (metadata) from the resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('iotevents:UntagResource');
        return this;
    }
    /**
     * Updates a detector model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateDetectorModel.html
     */
    updateDetectorModel() {
        this.add('iotevents:UpdateDetectorModel');
        return this;
    }
    /**
     * Updates an input.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInput.html
     */
    updateInput() {
        this.add('iotevents:UpdateInput');
        return this;
    }
    /**
     * Updates input routing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInputRouting.html
     */
    updateInputRouting() {
        this.add('iotevents:UpdateInputRouting');
        return this;
    }
    /**
     * Adds a resource of type detectorModel to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param detectorModelName - Identifier for the detectorModelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetectorModel(detectorModelName, account, region, partition) {
        var arn = 'arn:${Partition}:iotevents:${Region}:${Account}:detectorModel/${DetectorModelName}';
        arn = arn.replace('${DetectorModelName}', detectorModelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type input to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param inputName - Identifier for the inputName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInput(inputName, account, region, partition) {
        var arn = 'arn:${Partition}:iotevents:${Region}:${Account}:input/${InputName}';
        arn = arn.replace('${InputName}', inputName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * A tag key that is present in the request that the user makes to IoT Events.
     *
     * Applies to actions:
     * - .createDetectorModel()
     * - .createInput()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The tag key by which a tag value is attached to an IoT Events resource.
     *
     * Applies to resource types:
     * - detectorModel
     * - input
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The list of all the tag key names associated with the IoT Events resource in the request.
     *
     * Applies to actions:
     * - .createDetectorModel()
     * - .createInput()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Iotevents = Iotevents;
//# sourceMappingURL=data:application/json;base64,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