"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Greengrass = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [greengrass](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotgreengrass.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Greengrass extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [greengrass](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotgreengrass.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'greengrass';
        this.actionList = {
            "AssociateRoleToGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/associateroletogroup-put.html",
                "description": "Grants permission to associate a role with a group. The role's permissions must allow Greengrass core Lambda functions and connectors to perform actions in other AWS services.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "AssociateServiceRoleToAccount": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/associateserviceroletoaccount-put.html",
                "description": "Grants permission to associate a role with your account. AWS IoT Greengrass uses this role to access your Lambda functions and AWS IoT resources.",
                "accessLevel": "Permissions management"
            },
            "CreateConnectorDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinition-post.html",
                "description": "Grants permission to create a connector definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateConnectorDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing connector definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "CreateCoreDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinition-post.html",
                "description": "Grants permission to create a core definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCoreDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing core definition. Greengrass groups must each contain exactly one Greengrass core.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "CreateDeployment": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createdeployment-post.html",
                "description": "Grants permission to create a deployment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateDeviceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinition-post.html",
                "description": "Grants permission to create a device definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDeviceDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing device definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "CreateFunctionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinition-post.html",
                "description": "Grants permission to create a Lambda function definition to be used in a group that contains a list of Lambda functions and their configurations.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFunctionDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing Lambda function definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/creategroup-post.html",
                "description": "Grants permission to create a group.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGroupCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupcertificateauthority-post.html",
                "description": "Grants permission to create a CA for the group, or rotate the existing CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateGroupVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupversion-post.html",
                "description": "Grants permission to create a version of a group that has already been defined.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateLoggerDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinition-post.html",
                "description": "Grants permission to create a logger definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoggerDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing logger definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "CreateResourceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinition-post.html",
                "description": "Grants permission to create a resource definition that contains a list of resources to be used in a group.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateResourceDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing resource definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "CreateSoftwareUpdateJob": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createsoftwareupdatejob-post.html",
                "description": "Grants permission to create an AWS IoT job that will trigger your Greengrass cores to update the software they are running.",
                "accessLevel": "Write"
            },
            "CreateSubscriptionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinition-post.html",
                "description": "Grants permission to create a subscription definition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSubscriptionDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinitionversion-post.html",
                "description": "Grants permission to create a version of an existing subscription definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteConnectorDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deleteconnectordefinition-delete.html",
                "description": "Grants permission to delete a connector definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteCoreDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletecoredefinition-delete.html",
                "description": "Grants permission to delete a core definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteDeviceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletedevicedefinition-delete.html",
                "description": "Grants permission to delete a device definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteFunctionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletefunctiondefinition-delete.html",
                "description": "Grants permission to delete a Lambda function definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletegroup-delete.html",
                "description": "Grants permission to delete a group that is not currently in use in a deployment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteLoggerDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deleteloggerdefinition-delete.html",
                "description": "Grants permission to delete a logger definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteResourceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deleteresourcedefinition-delete.html",
                "description": "Grants permission to delete a resource definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "DeleteSubscriptionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/deletesubscriptiondefinition-delete.html",
                "description": "Grants permission to delete a subscription definition. Deleting a definition that is currently in use in a deployment affects future deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            },
            "DisassociateRoleFromGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/disassociaterolefromgroup-delete.html",
                "description": "Grants permission to disassociate the role from a group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DisassociateServiceRoleFromAccount": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/disassociateservicerolefromaccount-delete.html",
                "description": "Grants permission to disassociate the service role from an account. Without a service role, deployments will not work.",
                "accessLevel": "Write"
            },
            "GetAssociatedRole": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getassociatedrole-get.html",
                "description": "Grants permission to retrieve the role associated with a group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetBulkDeploymentStatus": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getbulkdeploymentstatus-get.html",
                "description": "Grants permission to return the status of a bulk deployment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": true
                    }
                }
            },
            "GetConnectivityInfo": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectivityinfo-get.html",
                "description": "Grants permission to retrieve the connectivity information for a core.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectivityInfo": {
                        "required": true
                    }
                }
            },
            "GetConnectorDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinition-get.html",
                "description": "Grants permission to retrieve information about a connector definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "GetConnectorDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a connector definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    },
                    "connectorDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetCoreDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinition-get.html",
                "description": "Grants permission to retrieve information about a core definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "GetCoreDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a core definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    },
                    "coreDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetDeploymentStatus": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getdeploymentstatus-get.html",
                "description": "Grants permission to return the status of a deployment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deployment": {
                        "required": true
                    },
                    "group": {
                        "required": true
                    }
                }
            },
            "GetDeviceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinition-get.html",
                "description": "Grants permission to retrieve information about a device definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "GetDeviceDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a device definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    },
                    "deviceDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetFunctionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinition-get.html",
                "description": "Grants permission to retrieve information about a Lambda function definition, such as its creation time and latest version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "GetFunctionDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a Lambda function definition version, such as which Lambda functions are included in the version and their configurations.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    },
                    "functionDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getgroup-get.html",
                "description": "Grants permission to retrieve information about a group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateauthority-get.html",
                "description": "Grants permission to return the public key of the CA associated with a group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificateAuthority": {
                        "required": true
                    },
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupCertificateConfiguration": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateconfiguration-get.html",
                "description": "Grants permission to retrieve the current configuration for the CA used by a group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupversion-get.html",
                "description": "Grants permission to retrieve information about a group version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    },
                    "groupVersion": {
                        "required": true
                    }
                }
            },
            "GetLoggerDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinition-get.html",
                "description": "Grants permission to retrieve information about a logger definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "GetLoggerDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a logger definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    },
                    "loggerDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetResourceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinition-get.html",
                "description": "Grants permission to retrieve information about a resource definition, such as its creation time and latest version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "GetResourceDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a resource definition version, such as which resources are included in the version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    },
                    "resourceDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "GetServiceRoleForAccount": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getserviceroleforaccount-get.html",
                "description": "Grants permission to retrieve the service role that is attached to an account.",
                "accessLevel": "Read"
            },
            "GetSubscriptionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinition-get.html",
                "description": "Grants permission to retrieve information about a subscription definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            },
            "GetSubscriptionDefinitionVersion": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinitionversion-get.html",
                "description": "Grants permission to retrieve information about a subscription definition version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    },
                    "subscriptionDefinitionVersion": {
                        "required": true
                    }
                }
            },
            "ListBulkDeploymentDetailedReports": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeploymentdetailedreports-get.html",
                "description": "Grants permission to retrieve a paginated list of the deployments that have been started in a bulk deployment operation and their current deployment status.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": true
                    }
                }
            },
            "ListBulkDeployments": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeployments-get.html",
                "description": "Grants permission to retrieve a list of bulk deployments.",
                "accessLevel": "List"
            },
            "ListConnectorDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitionversions-get.html",
                "description": "Grants permission to list the versions of a connector definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "ListConnectorDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitions-get.html",
                "description": "Grants permission to retrieve a list of connector definitions.",
                "accessLevel": "List"
            },
            "ListCoreDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitionversions-get.html",
                "description": "Grants permission to list the versions of a core definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "ListCoreDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitions-get.html",
                "description": "Grants permission to retrieve a list of core definitions.",
                "accessLevel": "List"
            },
            "ListDeployments": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listdeployments-get.html",
                "description": "Grants permission to retrieve a list of all deployments for a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListDeviceDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitionversions-get.html",
                "description": "Grants permission to list the versions of a device definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "ListDeviceDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitions-get.html",
                "description": "Grants permission to retrieve a list of device definitions.",
                "accessLevel": "List"
            },
            "ListFunctionDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitionversions-get.html",
                "description": "Grants permission to list the versions of a Lambda function definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "ListFunctionDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitions-get.html",
                "description": "Grants permission to retrieve a list of Lambda function definitions.",
                "accessLevel": "List"
            },
            "ListGroupCertificateAuthorities": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupcertificateauthorities-get.html",
                "description": "Grants permission to retrieve a list of current CAs for a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroupVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupversions-get.html",
                "description": "Grants permission to list the versions of a group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listgroups-get.html",
                "description": "Grants permission to retrieve a list of groups.",
                "accessLevel": "List"
            },
            "ListLoggerDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitionversions-get.html",
                "description": "Grants permission to list the versions of a logger definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "ListLoggerDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitions-get.html",
                "description": "Grants permission to retrieve a list of logger definitions.",
                "accessLevel": "List"
            },
            "ListResourceDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitionversions-get.html",
                "description": "Grants permission to list the versions of a resource definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "ListResourceDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitions-get.html",
                "description": "Grants permission to retrieve a list of resource definitions.",
                "accessLevel": "List"
            },
            "ListSubscriptionDefinitionVersions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitionversions-get.html",
                "description": "Grants permission to list the versions of a subscription definition.",
                "accessLevel": "List",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            },
            "ListSubscriptionDefinitions": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitions-get.html",
                "description": "Grants permission to retrieve a list of subscription definitions.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/listtagsforresource-get.html",
                "description": "Grants permission to list the tags for a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": false
                    },
                    "connectorDefinition": {
                        "required": false
                    },
                    "coreDefinition": {
                        "required": false
                    },
                    "deviceDefinition": {
                        "required": false
                    },
                    "functionDefinition": {
                        "required": false
                    },
                    "group": {
                        "required": false
                    },
                    "loggerDefinition": {
                        "required": false
                    },
                    "resourceDefinition": {
                        "required": false
                    },
                    "subscriptionDefinition": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ResetDeployments": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/resetdeployments-post.html",
                "description": "Grants permission to reset a group's deployments.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "StartBulkDeployment": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/startbulkdeployment-post.html",
                "description": "Grants permission to deploy multiple groups in one operation.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StopBulkDeployment": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/stopbulkdeployment-put.html",
                "description": "Grants permission to stop the execution of a bulk deployment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/tagresource-post.html",
                "description": "Grants permission to add tags to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": false
                    },
                    "connectorDefinition": {
                        "required": false
                    },
                    "coreDefinition": {
                        "required": false
                    },
                    "deviceDefinition": {
                        "required": false
                    },
                    "functionDefinition": {
                        "required": false
                    },
                    "group": {
                        "required": false
                    },
                    "loggerDefinition": {
                        "required": false
                    },
                    "resourceDefinition": {
                        "required": false
                    },
                    "subscriptionDefinition": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/untagresource-delete.html",
                "description": "Grants permission to remove tags from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bulkDeployment": {
                        "required": false
                    },
                    "connectorDefinition": {
                        "required": false
                    },
                    "coreDefinition": {
                        "required": false
                    },
                    "deviceDefinition": {
                        "required": false
                    },
                    "functionDefinition": {
                        "required": false
                    },
                    "group": {
                        "required": false
                    },
                    "loggerDefinition": {
                        "required": false
                    },
                    "resourceDefinition": {
                        "required": false
                    },
                    "subscriptionDefinition": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateConnectivityInfo": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectivityinfo-put.html",
                "description": "Grants permission to update the connectivity information for a Greengrass core. Any devices that belong to the group that has this core will receive this information in order to find the location of the core and connect to it.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectivityInfo": {
                        "required": true
                    }
                }
            },
            "UpdateConnectorDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectordefinition-put.html",
                "description": "Grants permission to update a connector definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectorDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateCoreDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updatecoredefinition-put.html",
                "description": "Grants permission to update a core definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "coreDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateDeviceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updatedevicedefinition-put.html",
                "description": "Grants permission to update a device definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deviceDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateFunctionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updatefunctiondefinition-put.html",
                "description": "Grants permission to update a Lambda function definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "functionDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updategroup-put.html",
                "description": "Grants permission to update a group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateGroupCertificateConfiguration": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updategroupcertificateconfiguration-put.html",
                "description": "Grants permission to update the certificate expiry time for a group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateLoggerDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updateloggerdefinition-put.html",
                "description": "Grants permission to update a logger definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "loggerDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateResourceDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updateresourcedefinition-put.html",
                "description": "Grants permission to update a resource definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "resourceDefinition": {
                        "required": true
                    }
                }
            },
            "UpdateSubscriptionDefinition": {
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/updatesubscriptiondefinition-put.html",
                "description": "Grants permission to update a subscription definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subscriptionDefinition": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "connectivityInfo": {
                "name": "connectivityInfo",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectivityinfo.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/things/${ThingName}/connectivityInfo",
                "conditionKeys": []
            },
            "artifact": {
                "name": "artifact",
                "url": "",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}/artifacts/lambda/${ArtifactId}",
                "conditionKeys": []
            },
            "certificateAuthority": {
                "name": "certificateAuthority",
                "url": "https://docs.aws.amazon.com/greengrass/latest/developerguide/gg-sec.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/certificateauthorities/${CertificateAuthorityId}",
                "conditionKeys": []
            },
            "deployment": {
                "name": "deployment",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-createdeploymentrequest.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}",
                "conditionKeys": []
            },
            "bulkDeployment": {
                "name": "bulkDeployment",
                "url": "https://docs.aws.amazon.com/greengrass/latest/developerguide/bulk-deploy-cli.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/bulk/deployments/${BulkDeploymentId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupinformation.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "groupVersion": {
                "name": "groupVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "coreDefinition": {
                "name": "coreDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-core.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "coreDefinitionVersion": {
                "name": "coreDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-coredefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "deviceDefinition": {
                "name": "deviceDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-device.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "deviceDefinitionVersion": {
                "name": "deviceDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-devicedefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "functionDefinition": {
                "name": "functionDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-function.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "functionDefinitionVersion": {
                "name": "functionDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-functiondefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "subscriptionDefinition": {
                "name": "subscriptionDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscription.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "subscriptionDefinitionVersion": {
                "name": "subscriptionDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscriptiondefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "loggerDefinition": {
                "name": "loggerDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-logger.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "loggerDefinitionVersion": {
                "name": "loggerDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-loggerdefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "resourceDefinition": {
                "name": "resourceDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resource.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "resourceDefinitionVersion": {
                "name": "resourceDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resourcedefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            },
            "connectorDefinition": {
                "name": "connectorDefinition",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connector.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "connectorDefinitionVersion": {
                "name": "connectorDefinitionVersion",
                "url": "https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectordefinitionversion.html",
                "arn": "arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}/versions/${VersionId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate a role with a group. The role's permissions must allow Greengrass core Lambda functions and connectors to perform actions in other AWS services.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/associateroletogroup-put.html
     */
    associateRoleToGroup() {
        this.add('greengrass:AssociateRoleToGroup');
        return this;
    }
    /**
     * Grants permission to associate a role with your account. AWS IoT Greengrass uses this role to access your Lambda functions and AWS IoT resources.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/associateserviceroletoaccount-put.html
     */
    associateServiceRoleToAccount() {
        this.add('greengrass:AssociateServiceRoleToAccount');
        return this;
    }
    /**
     * Grants permission to create a connector definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinition-post.html
     */
    createConnectorDefinition() {
        this.add('greengrass:CreateConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createconnectordefinitionversion-post.html
     */
    createConnectorDefinitionVersion() {
        this.add('greengrass:CreateConnectorDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a core definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinition-post.html
     */
    createCoreDefinition() {
        this.add('greengrass:CreateCoreDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing core definition. Greengrass groups must each contain exactly one Greengrass core.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createcoredefinitionversion-post.html
     */
    createCoreDefinitionVersion() {
        this.add('greengrass:CreateCoreDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdeployment-post.html
     */
    createDeployment() {
        this.add('greengrass:CreateDeployment');
        return this;
    }
    /**
     * Grants permission to create a device definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinition-post.html
     */
    createDeviceDefinition() {
        this.add('greengrass:CreateDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing device definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createdevicedefinitionversion-post.html
     */
    createDeviceDefinitionVersion() {
        this.add('greengrass:CreateDeviceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a Lambda function definition to be used in a group that contains a list of Lambda functions and their configurations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinition-post.html
     */
    createFunctionDefinition() {
        this.add('greengrass:CreateFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing Lambda function definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createfunctiondefinitionversion-post.html
     */
    createFunctionDefinitionVersion() {
        this.add('greengrass:CreateFunctionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroup-post.html
     */
    createGroup() {
        this.add('greengrass:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a CA for the group, or rotate the existing CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupcertificateauthority-post.html
     */
    createGroupCertificateAuthority() {
        this.add('greengrass:CreateGroupCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to create a version of a group that has already been defined.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/creategroupversion-post.html
     */
    createGroupVersion() {
        this.add('greengrass:CreateGroupVersion');
        return this;
    }
    /**
     * Grants permission to create a logger definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinition-post.html
     */
    createLoggerDefinition() {
        this.add('greengrass:CreateLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing logger definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createloggerdefinitionversion-post.html
     */
    createLoggerDefinitionVersion() {
        this.add('greengrass:CreateLoggerDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create a resource definition that contains a list of resources to be used in a group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinition-post.html
     */
    createResourceDefinition() {
        this.add('greengrass:CreateResourceDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createresourcedefinitionversion-post.html
     */
    createResourceDefinitionVersion() {
        this.add('greengrass:CreateResourceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to create an AWS IoT job that will trigger your Greengrass cores to update the software they are running.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsoftwareupdatejob-post.html
     */
    createSoftwareUpdateJob() {
        this.add('greengrass:CreateSoftwareUpdateJob');
        return this;
    }
    /**
     * Grants permission to create a subscription definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinition-post.html
     */
    createSubscriptionDefinition() {
        this.add('greengrass:CreateSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to create a version of an existing subscription definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/createsubscriptiondefinitionversion-post.html
     */
    createSubscriptionDefinitionVersion() {
        this.add('greengrass:CreateSubscriptionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to delete a connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteconnectordefinition-delete.html
     */
    deleteConnectorDefinition() {
        this.add('greengrass:DeleteConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to delete a core definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletecoredefinition-delete.html
     */
    deleteCoreDefinition() {
        this.add('greengrass:DeleteCoreDefinition');
        return this;
    }
    /**
     * Grants permission to delete a device definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletedevicedefinition-delete.html
     */
    deleteDeviceDefinition() {
        this.add('greengrass:DeleteDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to delete a Lambda function definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletefunctiondefinition-delete.html
     */
    deleteFunctionDefinition() {
        this.add('greengrass:DeleteFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to delete a group that is not currently in use in a deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletegroup-delete.html
     */
    deleteGroup() {
        this.add('greengrass:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete a logger definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteloggerdefinition-delete.html
     */
    deleteLoggerDefinition() {
        this.add('greengrass:DeleteLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to delete a resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deleteresourcedefinition-delete.html
     */
    deleteResourceDefinition() {
        this.add('greengrass:DeleteResourceDefinition');
        return this;
    }
    /**
     * Grants permission to delete a subscription definition. Deleting a definition that is currently in use in a deployment affects future deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/deletesubscriptiondefinition-delete.html
     */
    deleteSubscriptionDefinition() {
        this.add('greengrass:DeleteSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to disassociate the role from a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/disassociaterolefromgroup-delete.html
     */
    disassociateRoleFromGroup() {
        this.add('greengrass:DisassociateRoleFromGroup');
        return this;
    }
    /**
     * Grants permission to disassociate the service role from an account. Without a service role, deployments will not work.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/disassociateservicerolefromaccount-delete.html
     */
    disassociateServiceRoleFromAccount() {
        this.add('greengrass:DisassociateServiceRoleFromAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the role associated with a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getassociatedrole-get.html
     */
    getAssociatedRole() {
        this.add('greengrass:GetAssociatedRole');
        return this;
    }
    /**
     * Grants permission to return the status of a bulk deployment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getbulkdeploymentstatus-get.html
     */
    getBulkDeploymentStatus() {
        this.add('greengrass:GetBulkDeploymentStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the connectivity information for a core.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectivityinfo-get.html
     */
    getConnectivityInfo() {
        this.add('greengrass:GetConnectivityInfo');
        return this;
    }
    /**
     * Grants permission to retrieve information about a connector definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinition-get.html
     */
    getConnectorDefinition() {
        this.add('greengrass:GetConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a connector definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getconnectordefinitionversion-get.html
     */
    getConnectorDefinitionVersion() {
        this.add('greengrass:GetConnectorDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a core definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinition-get.html
     */
    getCoreDefinition() {
        this.add('greengrass:GetCoreDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a core definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getcoredefinitionversion-get.html
     */
    getCoreDefinitionVersion() {
        this.add('greengrass:GetCoreDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to return the status of a deployment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdeploymentstatus-get.html
     */
    getDeploymentStatus() {
        this.add('greengrass:GetDeploymentStatus');
        return this;
    }
    /**
     * Grants permission to retrieve information about a device definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinition-get.html
     */
    getDeviceDefinition() {
        this.add('greengrass:GetDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a device definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getdevicedefinitionversion-get.html
     */
    getDeviceDefinitionVersion() {
        this.add('greengrass:GetDeviceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a Lambda function definition, such as its creation time and latest version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinition-get.html
     */
    getFunctionDefinition() {
        this.add('greengrass:GetFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a Lambda function definition version, such as which Lambda functions are included in the version and their configurations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getfunctiondefinitionversion-get.html
     */
    getFunctionDefinitionVersion() {
        this.add('greengrass:GetFunctionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroup-get.html
     */
    getGroup() {
        this.add('greengrass:GetGroup');
        return this;
    }
    /**
     * Grants permission to return the public key of the CA associated with a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateauthority-get.html
     */
    getGroupCertificateAuthority() {
        this.add('greengrass:GetGroupCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to retrieve the current configuration for the CA used by a group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupcertificateconfiguration-get.html
     */
    getGroupCertificateConfiguration() {
        this.add('greengrass:GetGroupCertificateConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve information about a group version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getgroupversion-get.html
     */
    getGroupVersion() {
        this.add('greengrass:GetGroupVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a logger definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinition-get.html
     */
    getLoggerDefinition() {
        this.add('greengrass:GetLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a logger definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getloggerdefinitionversion-get.html
     */
    getLoggerDefinitionVersion() {
        this.add('greengrass:GetLoggerDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about a resource definition, such as its creation time and latest version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinition-get.html
     */
    getResourceDefinition() {
        this.add('greengrass:GetResourceDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a resource definition version, such as which resources are included in the version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getresourcedefinitionversion-get.html
     */
    getResourceDefinitionVersion() {
        this.add('greengrass:GetResourceDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve the service role that is attached to an account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getserviceroleforaccount-get.html
     */
    getServiceRoleForAccount() {
        this.add('greengrass:GetServiceRoleForAccount');
        return this;
    }
    /**
     * Grants permission to retrieve information about a subscription definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinition-get.html
     */
    getSubscriptionDefinition() {
        this.add('greengrass:GetSubscriptionDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve information about a subscription definition version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/getsubscriptiondefinitionversion-get.html
     */
    getSubscriptionDefinitionVersion() {
        this.add('greengrass:GetSubscriptionDefinitionVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a paginated list of the deployments that have been started in a bulk deployment operation and their current deployment status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeploymentdetailedreports-get.html
     */
    listBulkDeploymentDetailedReports() {
        this.add('greengrass:ListBulkDeploymentDetailedReports');
        return this;
    }
    /**
     * Grants permission to retrieve a list of bulk deployments.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listbulkdeployments-get.html
     */
    listBulkDeployments() {
        this.add('greengrass:ListBulkDeployments');
        return this;
    }
    /**
     * Grants permission to list the versions of a connector definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitionversions-get.html
     */
    listConnectorDefinitionVersions() {
        this.add('greengrass:ListConnectorDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of connector definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listconnectordefinitions-get.html
     */
    listConnectorDefinitions() {
        this.add('greengrass:ListConnectorDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a core definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitionversions-get.html
     */
    listCoreDefinitionVersions() {
        this.add('greengrass:ListCoreDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of core definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listcoredefinitions-get.html
     */
    listCoreDefinitions() {
        this.add('greengrass:ListCoreDefinitions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all deployments for a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdeployments-get.html
     */
    listDeployments() {
        this.add('greengrass:ListDeployments');
        return this;
    }
    /**
     * Grants permission to list the versions of a device definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitionversions-get.html
     */
    listDeviceDefinitionVersions() {
        this.add('greengrass:ListDeviceDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of device definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listdevicedefinitions-get.html
     */
    listDeviceDefinitions() {
        this.add('greengrass:ListDeviceDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a Lambda function definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitionversions-get.html
     */
    listFunctionDefinitionVersions() {
        this.add('greengrass:ListFunctionDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of Lambda function definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listfunctiondefinitions-get.html
     */
    listFunctionDefinitions() {
        this.add('greengrass:ListFunctionDefinitions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of current CAs for a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupcertificateauthorities-get.html
     */
    listGroupCertificateAuthorities() {
        this.add('greengrass:ListGroupCertificateAuthorities');
        return this;
    }
    /**
     * Grants permission to list the versions of a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroupversions-get.html
     */
    listGroupVersions() {
        this.add('greengrass:ListGroupVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listgroups-get.html
     */
    listGroups() {
        this.add('greengrass:ListGroups');
        return this;
    }
    /**
     * Grants permission to list the versions of a logger definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitionversions-get.html
     */
    listLoggerDefinitionVersions() {
        this.add('greengrass:ListLoggerDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of logger definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listloggerdefinitions-get.html
     */
    listLoggerDefinitions() {
        this.add('greengrass:ListLoggerDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a resource definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitionversions-get.html
     */
    listResourceDefinitionVersions() {
        this.add('greengrass:ListResourceDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of resource definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listresourcedefinitions-get.html
     */
    listResourceDefinitions() {
        this.add('greengrass:ListResourceDefinitions');
        return this;
    }
    /**
     * Grants permission to list the versions of a subscription definition.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitionversions-get.html
     */
    listSubscriptionDefinitionVersions() {
        this.add('greengrass:ListSubscriptionDefinitionVersions');
        return this;
    }
    /**
     * Grants permission to retrieve a list of subscription definitions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listsubscriptiondefinitions-get.html
     */
    listSubscriptionDefinitions() {
        this.add('greengrass:ListSubscriptionDefinitions');
        return this;
    }
    /**
     * Grants permission to list the tags for a resource.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/listtagsforresource-get.html
     */
    listTagsForResource() {
        this.add('greengrass:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to reset a group's deployments.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/resetdeployments-post.html
     */
    resetDeployments() {
        this.add('greengrass:ResetDeployments');
        return this;
    }
    /**
     * Grants permission to deploy multiple groups in one operation.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/startbulkdeployment-post.html
     */
    startBulkDeployment() {
        this.add('greengrass:StartBulkDeployment');
        return this;
    }
    /**
     * Grants permission to stop the execution of a bulk deployment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/stopbulkdeployment-put.html
     */
    stopBulkDeployment() {
        this.add('greengrass:StopBulkDeployment');
        return this;
    }
    /**
     * Grants permission to add tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/tagresource-post.html
     */
    tagResource() {
        this.add('greengrass:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/untagresource-delete.html
     */
    untagResource() {
        this.add('greengrass:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the connectivity information for a Greengrass core. Any devices that belong to the group that has this core will receive this information in order to find the location of the core and connect to it.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectivityinfo-put.html
     */
    updateConnectivityInfo() {
        this.add('greengrass:UpdateConnectivityInfo');
        return this;
    }
    /**
     * Grants permission to update a connector definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateconnectordefinition-put.html
     */
    updateConnectorDefinition() {
        this.add('greengrass:UpdateConnectorDefinition');
        return this;
    }
    /**
     * Grants permission to update a core definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatecoredefinition-put.html
     */
    updateCoreDefinition() {
        this.add('greengrass:UpdateCoreDefinition');
        return this;
    }
    /**
     * Grants permission to update a device definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatedevicedefinition-put.html
     */
    updateDeviceDefinition() {
        this.add('greengrass:UpdateDeviceDefinition');
        return this;
    }
    /**
     * Grants permission to update a Lambda function definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatefunctiondefinition-put.html
     */
    updateFunctionDefinition() {
        this.add('greengrass:UpdateFunctionDefinition');
        return this;
    }
    /**
     * Grants permission to update a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updategroup-put.html
     */
    updateGroup() {
        this.add('greengrass:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to update the certificate expiry time for a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updategroupcertificateconfiguration-put.html
     */
    updateGroupCertificateConfiguration() {
        this.add('greengrass:UpdateGroupCertificateConfiguration');
        return this;
    }
    /**
     * Grants permission to update a logger definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateloggerdefinition-put.html
     */
    updateLoggerDefinition() {
        this.add('greengrass:UpdateLoggerDefinition');
        return this;
    }
    /**
     * Grants permission to update a resource definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updateresourcedefinition-put.html
     */
    updateResourceDefinition() {
        this.add('greengrass:UpdateResourceDefinition');
        return this;
    }
    /**
     * Grants permission to update a subscription definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/updatesubscriptiondefinition-put.html
     */
    updateSubscriptionDefinition() {
        this.add('greengrass:UpdateSubscriptionDefinition');
        return this;
    }
    /**
     * Adds a resource of type connectivityInfo to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectivityinfo.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnectivityInfo(thingName, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/things/${ThingName}/connectivityInfo';
        arn = arn.replace('${ThingName}', thingName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type artifact to the statement
     *
     * @param groupId - Identifier for the groupId.
     * @param deploymentId - Identifier for the deploymentId.
     * @param artifactId - Identifier for the artifactId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onArtifact(groupId, deploymentId, artifactId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}/artifacts/lambda/${ArtifactId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${DeploymentId}', deploymentId);
        arn = arn.replace('${ArtifactId}', artifactId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type certificateAuthority to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/developerguide/gg-sec.html
     *
     * @param groupId - Identifier for the groupId.
     * @param certificateAuthorityId - Identifier for the certificateAuthorityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCertificateAuthority(groupId, certificateAuthorityId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/certificateauthorities/${CertificateAuthorityId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${CertificateAuthorityId}', certificateAuthorityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deployment to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-createdeploymentrequest.html
     *
     * @param groupId - Identifier for the groupId.
     * @param deploymentId - Identifier for the deploymentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeployment(groupId, deploymentId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/deployments/${DeploymentId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${DeploymentId}', deploymentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bulkDeployment to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/developerguide/bulk-deploy-cli.html
     *
     * @param bulkDeploymentId - Identifier for the bulkDeploymentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBulkDeployment(bulkDeploymentId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/bulk/deployments/${BulkDeploymentId}';
        arn = arn.replace('${BulkDeploymentId}', bulkDeploymentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupinformation.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type groupVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-groupversion.html
     *
     * @param groupId - Identifier for the groupId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroupVersion(groupId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/groups/${GroupId}/versions/${VersionId}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type coreDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-core.html
     *
     * @param coreDefinitionId - Identifier for the coreDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCoreDefinition(coreDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}';
        arn = arn.replace('${CoreDefinitionId}', coreDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type coreDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-coredefinitionversion.html
     *
     * @param coreDefinitionId - Identifier for the coreDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCoreDefinitionVersion(coreDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/cores/${CoreDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${CoreDefinitionId}', coreDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deviceDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-device.html
     *
     * @param deviceDefinitionId - Identifier for the deviceDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeviceDefinition(deviceDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}';
        arn = arn.replace('${DeviceDefinitionId}', deviceDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deviceDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-devicedefinitionversion.html
     *
     * @param deviceDefinitionId - Identifier for the deviceDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeviceDefinitionVersion(deviceDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/devices/${DeviceDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${DeviceDefinitionId}', deviceDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type functionDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-function.html
     *
     * @param functionDefinitionId - Identifier for the functionDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFunctionDefinition(functionDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}';
        arn = arn.replace('${FunctionDefinitionId}', functionDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type functionDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-functiondefinitionversion.html
     *
     * @param functionDefinitionId - Identifier for the functionDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunctionDefinitionVersion(functionDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/functions/${FunctionDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${FunctionDefinitionId}', functionDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subscriptionDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscription.html
     *
     * @param subscriptionDefinitionId - Identifier for the subscriptionDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubscriptionDefinition(subscriptionDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}';
        arn = arn.replace('${SubscriptionDefinitionId}', subscriptionDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subscriptionDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-subscriptiondefinitionversion.html
     *
     * @param subscriptionDefinitionId - Identifier for the subscriptionDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSubscriptionDefinitionVersion(subscriptionDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/subscriptions/${SubscriptionDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${SubscriptionDefinitionId}', subscriptionDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loggerDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-logger.html
     *
     * @param loggerDefinitionId - Identifier for the loggerDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLoggerDefinition(loggerDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}';
        arn = arn.replace('${LoggerDefinitionId}', loggerDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loggerDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-loggerdefinitionversion.html
     *
     * @param loggerDefinitionId - Identifier for the loggerDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoggerDefinitionVersion(loggerDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/loggers/${LoggerDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${LoggerDefinitionId}', loggerDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resourceDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resource.html
     *
     * @param resourceDefinitionId - Identifier for the resourceDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResourceDefinition(resourceDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}';
        arn = arn.replace('${ResourceDefinitionId}', resourceDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type resourceDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-resourcedefinitionversion.html
     *
     * @param resourceDefinitionId - Identifier for the resourceDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onResourceDefinitionVersion(resourceDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/resources/${ResourceDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${ResourceDefinitionId}', resourceDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connectorDefinition to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connector.html
     *
     * @param connectorDefinitionId - Identifier for the connectorDefinitionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectorDefinition(connectorDefinitionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}';
        arn = arn.replace('${ConnectorDefinitionId}', connectorDefinitionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connectorDefinitionVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/latest/apireference/definitions-connectordefinitionversion.html
     *
     * @param connectorDefinitionId - Identifier for the connectorDefinitionId.
     * @param versionId - Identifier for the versionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnectorDefinitionVersion(connectorDefinitionId, versionId, account, region, partition) {
        var arn = 'arn:${Partition}:greengrass:${Region}:${Account}:/greengrass/definition/connectors/${ConnectorDefinitionId}/versions/${VersionId}';
        arn = arn.replace('${ConnectorDefinitionId}', connectorDefinitionId);
        arn = arn.replace('${VersionId}', versionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by checking date/time conditions for the current date and time.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifAwsCurrentTime(value, operator) {
        if (typeof value.getMonth === "function") {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === "function") {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if(`aws:CurrentTime`, value, operator || 'DateEquals');
    }
    /**
     * Filters access by checking date/time conditions for the current date and time in epoch or Unix time.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifAwsEpochTime(value, operator) {
        if (typeof value.getMonth === "function") {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === "function") {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if(`aws:EpochTime`, value, operator || 'DateEquals');
    }
    /**
     * Filters access by checking how long ago (in seconds) the security credentials validated by multi-factor authentication (MFA) in the request were issued using MFA.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifAwsMultiFactorAuthAge(value, operator) {
        return this.if(`aws:MultiFactorAuthAge`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by checking whether multi-factor authentication (MFA) was used to validate the temporary security credentials that made the current request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAwsMultiFactorAuthPresent(value) {
        return this.if(`aws:MultiFactorAuthPresent`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters create requests based on the allowed set of values for each of the mandatory tags.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .createConnectorDefinition()
     * - .createCoreDefinition()
     * - .createDeviceDefinition()
     * - .createFunctionDefinition()
     * - .createGroup()
     * - .createLoggerDefinition()
     * - .createResourceDefinition()
     * - .createSubscriptionDefinition()
     * - .listTagsForResource()
     * - .startBulkDeployment()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag value associated with the resource.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to resource types:
     * - bulkDeployment
     * - group
     * - coreDefinition
     * - deviceDefinition
     * - functionDefinition
     * - subscriptionDefinition
     * - loggerDefinition
     * - resourceDefinition
     * - connectorDefinition
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by checking whether the request was sent using SSL.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAwsSecureTransport(value) {
        return this.if(`aws:SecureTransport`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters create requests based on the presence of mandatory tags in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .createConnectorDefinition()
     * - .createCoreDefinition()
     * - .createDeviceDefinition()
     * - .createFunctionDefinition()
     * - .createGroup()
     * - .createLoggerDefinition()
     * - .createResourceDefinition()
     * - .createSubscriptionDefinition()
     * - .listTagsForResource()
     * - .startBulkDeployment()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the requester's client application.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserAgent(value, operator) {
        return this.if(`aws:UserAgent`, value, operator || 'StringLike');
    }
}
exports.Greengrass = Greengrass;
//# sourceMappingURL=data:application/json;base64,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