"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mobilehub = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mobilehub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmobilehub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mobilehub extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mobilehub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmobilehub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mobilehub';
        this.actionList = {
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Create a project",
                "accessLevel": "Write"
            },
            "CreateServiceRole": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Enable AWS Mobile Hub in the account by creating the required service role",
                "accessLevel": "Write"
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Delete the specified project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteProjectSnapshot": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Delete a saved snapshot of project configuration",
                "accessLevel": "Write"
            },
            "DeployToStage": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Deploy changes to the specified stage",
                "accessLevel": "Write"
            },
            "DescribeBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Describe the download bundle",
                "accessLevel": "Read"
            },
            "ExportBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Export the download bundle",
                "accessLevel": "Read"
            },
            "ExportProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Export the project configuration",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GenerateProjectParameters": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Generate project parameters required for code generation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GetProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Get project configuration and resources",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "GetProjectSnapshot": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Fetch the previously exported project configuration snapshot",
                "accessLevel": "Read"
            },
            "ImportProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Create a new project from the previously exported project configuration",
                "accessLevel": "Write"
            },
            "InstallBundle": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Install a bundle in the project deployments S3 bucket",
                "accessLevel": "Write"
            },
            "ListAvailableConnectors": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List the available SaaS (Software as a Service) connectors",
                "accessLevel": "List"
            },
            "ListAvailableFeatures": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List available features",
                "accessLevel": "List"
            },
            "ListAvailableRegions": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List available regions for projects",
                "accessLevel": "List"
            },
            "ListBundles": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List the available download bundles",
                "accessLevel": "List"
            },
            "ListProjectSnapshots": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List saved snapshots of project configuration",
                "accessLevel": "List"
            },
            "ListProjects": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "List projects",
                "accessLevel": "List"
            },
            "SynchronizeProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Synchronize state of resources into project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Update project",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ValidateProject": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Validate a mobile hub project.",
                "accessLevel": "Read"
            },
            "VerifyServiceRole": {
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html",
                "description": "Verify AWS Mobile Hub is enabled in the account",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "project": {
                "name": "project",
                "url": "https://docs.aws.amazon.com/mobile-hub/latest/developerguide/reference-mobile-hub-iam-managed-policies.html",
                "arn": "arn:${Partition}:mobilehub:${Region}:${Account}:project/${ProjectId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Create a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    createProject() {
        this.add('mobilehub:CreateProject');
        return this;
    }
    /**
     * Enable AWS Mobile Hub in the account by creating the required service role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    createServiceRole() {
        this.add('mobilehub:CreateServiceRole');
        return this;
    }
    /**
     * Delete the specified project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    deleteProject() {
        this.add('mobilehub:DeleteProject');
        return this;
    }
    /**
     * Delete a saved snapshot of project configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    deleteProjectSnapshot() {
        this.add('mobilehub:DeleteProjectSnapshot');
        return this;
    }
    /**
     * Deploy changes to the specified stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    deployToStage() {
        this.add('mobilehub:DeployToStage');
        return this;
    }
    /**
     * Describe the download bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    describeBundle() {
        this.add('mobilehub:DescribeBundle');
        return this;
    }
    /**
     * Export the download bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    exportBundle() {
        this.add('mobilehub:ExportBundle');
        return this;
    }
    /**
     * Export the project configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    exportProject() {
        this.add('mobilehub:ExportProject');
        return this;
    }
    /**
     * Generate project parameters required for code generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    generateProjectParameters() {
        this.add('mobilehub:GenerateProjectParameters');
        return this;
    }
    /**
     * Get project configuration and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    getProject() {
        this.add('mobilehub:GetProject');
        return this;
    }
    /**
     * Fetch the previously exported project configuration snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    getProjectSnapshot() {
        this.add('mobilehub:GetProjectSnapshot');
        return this;
    }
    /**
     * Create a new project from the previously exported project configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    importProject() {
        this.add('mobilehub:ImportProject');
        return this;
    }
    /**
     * Install a bundle in the project deployments S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    installBundle() {
        this.add('mobilehub:InstallBundle');
        return this;
    }
    /**
     * List the available SaaS (Software as a Service) connectors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listAvailableConnectors() {
        this.add('mobilehub:ListAvailableConnectors');
        return this;
    }
    /**
     * List available features
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listAvailableFeatures() {
        this.add('mobilehub:ListAvailableFeatures');
        return this;
    }
    /**
     * List available regions for projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listAvailableRegions() {
        this.add('mobilehub:ListAvailableRegions');
        return this;
    }
    /**
     * List the available download bundles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listBundles() {
        this.add('mobilehub:ListBundles');
        return this;
    }
    /**
     * List saved snapshots of project configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listProjectSnapshots() {
        this.add('mobilehub:ListProjectSnapshots');
        return this;
    }
    /**
     * List projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    listProjects() {
        this.add('mobilehub:ListProjects');
        return this;
    }
    /**
     * Synchronize state of resources into project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    synchronizeProject() {
        this.add('mobilehub:SynchronizeProject');
        return this;
    }
    /**
     * Update project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    updateProject() {
        this.add('mobilehub:UpdateProject');
        return this;
    }
    /**
     * Validate a mobile hub project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    validateProject() {
        this.add('mobilehub:ValidateProject');
        return this;
    }
    /**
     * Verify AWS Mobile Hub is enabled in the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/managed-policies.html
     */
    verifyServiceRole() {
        this.add('mobilehub:VerifyServiceRole');
        return this;
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/mobile-hub/latest/developerguide/reference-mobile-hub-iam-managed-policies.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectId, account, region, partition) {
        var arn = 'arn:${Partition}:mobilehub:${Region}:${Account}:project/${ProjectId}';
        arn = arn.replace('${ProjectId}', projectId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mobilehub = Mobilehub;
//# sourceMappingURL=data:application/json;base64,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