"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Secretsmanager = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [secretsmanager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecretsmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Secretsmanager extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [secretsmanager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecretsmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'secretsmanager';
        this.actionList = {
            "CancelRotateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to cancel an in-progress secret rotation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "CreateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to create a secret that stores encrypted data that can be queried and rotated.",
                "accessLevel": "Write",
                "conditions": [
                    "secretsmanager:Name",
                    "secretsmanager:Description",
                    "secretsmanager:KmsKeyId",
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to delete the resource policy attached to a secret.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DeleteSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to delete a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:RecoveryWindowInDays",
                    "secretsmanager:ForceDeleteWithoutRecovery",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DescribeSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to retrieve the metadata about a secret, but not the encrypted data.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "GetRandomPassword": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to generate a random string for use in password creation.",
                "accessLevel": "Read"
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to get the resource policy attached to a secret.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "GetSecretValue": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to retrieve and decrypt the encrypted data.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:VersionId",
                    "secretsmanager:VersionStage",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "ListSecretVersionIds": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to list the available versions of a secret.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "ListSecrets": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to list the available secrets.",
                "accessLevel": "List"
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to attach a resource policy to a secret.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "PutSecretValue": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to create a new version of the secret with new encrypted data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "RestoreSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to cancel deletion of a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "RotateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to start rotation of a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:RotationLambdaARN",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to add tags to a secret.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to remove tags from a secret.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "aws:TagKeys",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UpdateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to update a secret with new metadata or with a new version of the encrypted data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:Description",
                    "secretsmanager:KmsKeyId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UpdateSecretVersionStage": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to move a stage from one secret to another.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:VersionStage",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            }
        };
        this.resourceTypes = {
            "Secret": {
                "name": "Secret",
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-resources",
                "arn": "arn:${Partition}:secretsmanager:${Region}:${Account}:secret:${SecretId}",
                "conditionKeys": [
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:ResourceTag/tag-key",
                    "secretsmanager:resource/AllowRotationLambdaArn"
                ]
            }
        };
    }
    /**
     * Enables the user to cancel an in-progress secret rotation.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    cancelRotateSecret() {
        this.add('secretsmanager:CancelRotateSecret');
        return this;
    }
    /**
     * Enables the user to create a secret that stores encrypted data that can be queried and rotated.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifName()
     * - .ifDescription()
     * - .ifKmsKeyId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    createSecret() {
        this.add('secretsmanager:CreateSecret');
        return this;
    }
    /**
     * Enables the user to delete the resource policy attached to a secret.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    deleteResourcePolicy() {
        this.add('secretsmanager:DeleteResourcePolicy');
        return this;
    }
    /**
     * Enables the user to delete a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifRecoveryWindowInDays()
     * - .ifForceDeleteWithoutRecovery()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    deleteSecret() {
        this.add('secretsmanager:DeleteSecret');
        return this;
    }
    /**
     * Enables the user to retrieve the metadata about a secret, but not the encrypted data.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    describeSecret() {
        this.add('secretsmanager:DescribeSecret');
        return this;
    }
    /**
     * Enables the user to generate a random string for use in password creation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    getRandomPassword() {
        this.add('secretsmanager:GetRandomPassword');
        return this;
    }
    /**
     * Enables the user to get the resource policy attached to a secret.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    getResourcePolicy() {
        this.add('secretsmanager:GetResourcePolicy');
        return this;
    }
    /**
     * Enables the user to retrieve and decrypt the encrypted data.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifVersionId()
     * - .ifVersionStage()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    getSecretValue() {
        this.add('secretsmanager:GetSecretValue');
        return this;
    }
    /**
     * Enables the user to list the available versions of a secret.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    listSecretVersionIds() {
        this.add('secretsmanager:ListSecretVersionIds');
        return this;
    }
    /**
     * Enables the user to list the available secrets.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    listSecrets() {
        this.add('secretsmanager:ListSecrets');
        return this;
    }
    /**
     * Enables the user to attach a resource policy to a secret.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    putResourcePolicy() {
        this.add('secretsmanager:PutResourcePolicy');
        return this;
    }
    /**
     * Enables the user to create a new version of the secret with new encrypted data.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    putSecretValue() {
        this.add('secretsmanager:PutSecretValue');
        return this;
    }
    /**
     * Enables the user to cancel deletion of a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    restoreSecret() {
        this.add('secretsmanager:RestoreSecret');
        return this;
    }
    /**
     * Enables the user to start rotation of a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifRotationLambdaARN()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    rotateSecret() {
        this.add('secretsmanager:RotateSecret');
        return this;
    }
    /**
     * Enables the user to add tags to a secret.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    tagResource() {
        this.add('secretsmanager:TagResource');
        return this;
    }
    /**
     * Enables the user to remove tags from a secret.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifAwsTagKeys()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    untagResource() {
        this.add('secretsmanager:UntagResource');
        return this;
    }
    /**
     * Enables the user to update a secret with new metadata or with a new version of the encrypted data.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifDescription()
     * - .ifKmsKeyId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    updateSecret() {
        this.add('secretsmanager:UpdateSecret');
        return this;
    }
    /**
     * Enables the user to move a stage from one secret to another.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifVersionStage()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    updateSecretVersionStage() {
        this.add('secretsmanager:UpdateSecretVersionStage');
        return this;
    }
    /**
     * Adds a resource of type Secret to the statement
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-resources
     *
     * @param secretId - Identifier for the secretId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     * - .ifResource()
     */
    onSecret(secretId, account, region, partition) {
        var arn = 'arn:${Partition}:secretsmanager:${Region}:${Account}:secret:${SecretId}';
        arn = arn.replace('${SecretId}', secretId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the Secrets Manager service.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     * - .tagResource()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param tagkey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagkey, value, operator) {
        return this.if(`aws:RequestTag/${tagkey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key namespresent in the request the user makes to the Secrets Manager service.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     * - .tagResource()
     * - .untagResource()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the description text in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     * - .updateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDescription(value, operator) {
        return this.if(`secretsmanager:Description`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the secret is to be deleted immediately without any recovery window.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .deleteSecret()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifForceDeleteWithoutRecovery(value) {
        return this.if(`secretsmanager:ForceDeleteWithoutRecovery`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of the KMS key in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     * - .updateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKmsKeyId(value, operator) {
        return this.if(`secretsmanager:KmsKeyId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the friendly name of the secret in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`secretsmanager:Name`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the number of days that Secrets Manager waits before it can delete the secret.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .deleteSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRecoveryWindowInDays(value, operator) {
        return this.if(`secretsmanager:RecoveryWindowInDays`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by a tag key and value pair.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .cancelRotateSecret()
     * - .createSecret()
     * - .deleteResourcePolicy()
     * - .deleteSecret()
     * - .describeSecret()
     * - .getResourcePolicy()
     * - .getSecretValue()
     * - .listSecretVersionIds()
     * - .putResourcePolicy()
     * - .putSecretValue()
     * - .restoreSecret()
     * - .rotateSecret()
     * - .tagResource()
     * - .untagResource()
     * - .updateSecret()
     * - .updateSecretVersionStage()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param tagkey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagkey, value, operator) {
        return this.if(`secretsmanager:ResourceTag/${tagkey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the rotation Lambda function in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .rotateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifRotationLambdaARN(value, operator) {
        return this.if(`secretsmanager:RotationLambdaARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the SecretID value in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .cancelRotateSecret()
     * - .deleteResourcePolicy()
     * - .deleteSecret()
     * - .describeSecret()
     * - .getResourcePolicy()
     * - .getSecretValue()
     * - .listSecretVersionIds()
     * - .putResourcePolicy()
     * - .putSecretValue()
     * - .restoreSecret()
     * - .rotateSecret()
     * - .tagResource()
     * - .untagResource()
     * - .updateSecret()
     * - .updateSecretVersionStage()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSecretId(value, operator) {
        return this.if(`secretsmanager:SecretId`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the unique identifier of the version of the secret in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .getSecretValue()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value, operator) {
        return this.if(`secretsmanager:VersionId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the list of version stages in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .getSecretValue()
     * - .updateSecretVersionStage()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionStage(value, operator) {
        return this.if(`secretsmanager:VersionStage`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the rotation Lambda function associated with the secret.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .cancelRotateSecret()
     * - .deleteResourcePolicy()
     * - .deleteSecret()
     * - .describeSecret()
     * - .getResourcePolicy()
     * - .getSecretValue()
     * - .listSecretVersionIds()
     * - .putResourcePolicy()
     * - .putSecretValue()
     * - .restoreSecret()
     * - .rotateSecret()
     * - .tagResource()
     * - .untagResource()
     * - .updateSecret()
     * - .updateSecretVersionStage()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param allowRotationLambdaArn The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifResource(allowRotationLambdaArn, value, operator) {
        return this.if(`secretsmanager:resource/${allowRotationLambdaArn}`, value, operator || 'ArnEquals');
    }
}
exports.Secretsmanager = Secretsmanager;
//# sourceMappingURL=data:application/json;base64,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