"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Datapipeline = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [datapipeline](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_datapipeline.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Datapipeline extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [datapipeline](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_datapipeline.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'datapipeline';
        this.actionList = {
            "ActivatePipeline": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ActivatePipeline.html",
                "description": "Validates the specified pipeline and starts processing pipeline tasks. If the pipeline does not pass validation, activation fails.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            },
            "AddTags": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_AddTags.html",
                "description": "Adds or modifies tags for the specified pipeline.",
                "accessLevel": "Tagging",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "CreatePipeline": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_CreatePipeline.html",
                "description": "Creates a new, empty pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:Tag"
                ]
            },
            "DeactivatePipeline": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeactivatePipeline.html",
                "description": "Deactivates the specified running pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            },
            "DeletePipeline": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeletePipeline.html",
                "description": "Deletes a pipeline, its pipeline definition, and its run history.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "DescribeObjects": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribeObjects.html",
                "description": "Gets the object definitions for a set of objects associated with the pipeline.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "DescribePipelines": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribePipelines.html",
                "description": "Retrieves metadata about one or more pipelines.",
                "accessLevel": "List",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "EvaluateExpression": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_EvaluateExpression.html",
                "description": "Task runners call EvaluateExpression to evaluate a string in the context of the specified object.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "GetAccountLimits": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetAccountLimits.html",
                "description": "Description for GetAccountLimits",
                "accessLevel": "List"
            },
            "GetPipelineDefinition": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetPipelineDefinition.html",
                "description": "Gets the definition of the specified pipeline.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            },
            "ListPipelines": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ListPipelines.html",
                "description": "Lists the pipeline identifiers for all active pipelines that you have permission to access.",
                "accessLevel": "List"
            },
            "PollForTask": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PollForTask.html",
                "description": "Task runners call PollForTask to receive a task to perform from AWS Data Pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:workerGroup"
                ]
            },
            "PutAccountLimits": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutAccountLimits.html",
                "description": "Description for PutAccountLimits",
                "accessLevel": "Write"
            },
            "PutPipelineDefinition": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutPipelineDefinition.html",
                "description": "Adds tasks, schedules, and preconditions to the specified pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            },
            "QueryObjects": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_QueryObjects.html",
                "description": "Queries the specified pipeline for the names of objects that match the specified set of conditions.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_RemoveTags.html",
                "description": "Removes existing tags from the specified pipeline.",
                "accessLevel": "Tagging",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "ReportTaskProgress": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskProgress.html",
                "description": "Task runners call ReportTaskProgress when assigned a task to acknowledge that it has the task.",
                "accessLevel": "Write"
            },
            "ReportTaskRunnerHeartbeat": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskRunnerHeartbeat.html",
                "description": "Task runners call ReportTaskRunnerHeartbeat every 15 minutes to indicate that they are operational.",
                "accessLevel": "Write"
            },
            "SetStatus": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetStatus.html",
                "description": "Requests that the status of the specified physical or logical pipeline objects be updated in the specified pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "SetTaskStatus": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetTaskStatus.html",
                "description": "Task runners call SetTaskStatus to notify AWS Data Pipeline that a task is completed and provide information about the final status.",
                "accessLevel": "Write"
            },
            "ValidatePipelineDefinition": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ValidatePipelineDefinition.html",
                "description": "Validates the specified pipeline definition to ensure that it is well formed and can be run without error.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Validates the specified pipeline and starts processing pipeline tasks. If the pipeline does not pass validation, activation fails.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ActivatePipeline.html
     */
    activatePipeline() {
        this.add('datapipeline:ActivatePipeline');
        return this;
    }
    /**
     * Adds or modifies tags for the specified pipeline.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_AddTags.html
     */
    addTags() {
        this.add('datapipeline:AddTags');
        return this;
    }
    /**
     * Creates a new, empty pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_CreatePipeline.html
     */
    createPipeline() {
        this.add('datapipeline:CreatePipeline');
        return this;
    }
    /**
     * Deactivates the specified running pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeactivatePipeline.html
     */
    deactivatePipeline() {
        this.add('datapipeline:DeactivatePipeline');
        return this;
    }
    /**
     * Deletes a pipeline, its pipeline definition, and its run history.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeletePipeline.html
     */
    deletePipeline() {
        this.add('datapipeline:DeletePipeline');
        return this;
    }
    /**
     * Gets the object definitions for a set of objects associated with the pipeline.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribeObjects.html
     */
    describeObjects() {
        this.add('datapipeline:DescribeObjects');
        return this;
    }
    /**
     * Retrieves metadata about one or more pipelines.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribePipelines.html
     */
    describePipelines() {
        this.add('datapipeline:DescribePipelines');
        return this;
    }
    /**
     * Task runners call EvaluateExpression to evaluate a string in the context of the specified object.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_EvaluateExpression.html
     */
    evaluateExpression() {
        this.add('datapipeline:EvaluateExpression');
        return this;
    }
    /**
     * Description for GetAccountLimits
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetAccountLimits.html
     */
    getAccountLimits() {
        this.add('datapipeline:GetAccountLimits');
        return this;
    }
    /**
     * Gets the definition of the specified pipeline.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetPipelineDefinition.html
     */
    getPipelineDefinition() {
        this.add('datapipeline:GetPipelineDefinition');
        return this;
    }
    /**
     * Lists the pipeline identifiers for all active pipelines that you have permission to access.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ListPipelines.html
     */
    listPipelines() {
        this.add('datapipeline:ListPipelines');
        return this;
    }
    /**
     * Task runners call PollForTask to receive a task to perform from AWS Data Pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PollForTask.html
     */
    pollForTask() {
        this.add('datapipeline:PollForTask');
        return this;
    }
    /**
     * Description for PutAccountLimits
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutAccountLimits.html
     */
    putAccountLimits() {
        this.add('datapipeline:PutAccountLimits');
        return this;
    }
    /**
     * Adds tasks, schedules, and preconditions to the specified pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutPipelineDefinition.html
     */
    putPipelineDefinition() {
        this.add('datapipeline:PutPipelineDefinition');
        return this;
    }
    /**
     * Queries the specified pipeline for the names of objects that match the specified set of conditions.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_QueryObjects.html
     */
    queryObjects() {
        this.add('datapipeline:QueryObjects');
        return this;
    }
    /**
     * Removes existing tags from the specified pipeline.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_RemoveTags.html
     */
    removeTags() {
        this.add('datapipeline:RemoveTags');
        return this;
    }
    /**
     * Task runners call ReportTaskProgress when assigned a task to acknowledge that it has the task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskProgress.html
     */
    reportTaskProgress() {
        this.add('datapipeline:ReportTaskProgress');
        return this;
    }
    /**
     * Task runners call ReportTaskRunnerHeartbeat every 15 minutes to indicate that they are operational.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskRunnerHeartbeat.html
     */
    reportTaskRunnerHeartbeat() {
        this.add('datapipeline:ReportTaskRunnerHeartbeat');
        return this;
    }
    /**
     * Requests that the status of the specified physical or logical pipeline objects be updated in the specified pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetStatus.html
     */
    setStatus() {
        this.add('datapipeline:SetStatus');
        return this;
    }
    /**
     * Task runners call SetTaskStatus to notify AWS Data Pipeline that a task is completed and provide information about the final status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetTaskStatus.html
     */
    setTaskStatus() {
        this.add('datapipeline:SetTaskStatus');
        return this;
    }
    /**
     * Validates the specified pipeline definition to ensure that it is well formed and can be run without error.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ValidatePipelineDefinition.html
     */
    validatePipelineDefinition() {
        this.add('datapipeline:ValidatePipelineDefinition');
        return this;
    }
    /**
     * The IAM user that created the pipeline.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-example-tag-policies.html#ex3
     *
     * Applies to actions:
     * - .activatePipeline()
     * - .addTags()
     * - .deactivatePipeline()
     * - .deletePipeline()
     * - .describeObjects()
     * - .describePipelines()
     * - .evaluateExpression()
     * - .getPipelineDefinition()
     * - .putPipelineDefinition()
     * - .queryObjects()
     * - .removeTags()
     * - .setStatus()
     * - .validatePipelineDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPipelineCreator(value, operator) {
        return this.if(`datapipeline:PipelineCreator`, value, operator || 'ArnEquals');
    }
    /**
     * A customer-specified key/value pair that can be attached to a resource.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-resourcebased-access.html#dp-control-access-tags
     *
     * Applies to actions:
     * - .activatePipeline()
     * - .addTags()
     * - .createPipeline()
     * - .deactivatePipeline()
     * - .deletePipeline()
     * - .describeObjects()
     * - .describePipelines()
     * - .evaluateExpression()
     * - .getPipelineDefinition()
     * - .putPipelineDefinition()
     * - .queryObjects()
     * - .removeTags()
     * - .setStatus()
     * - .validatePipelineDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifTag(value, operator) {
        return this.if(`datapipeline:Tag`, value, operator || 'ArnEquals');
    }
    /**
     * The name of a worker group for which a Task Runner retrieves work.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-resourcebased-access.html#dp-control-access-workergroup
     *
     * Applies to actions:
     * - .activatePipeline()
     * - .deactivatePipeline()
     * - .getPipelineDefinition()
     * - .pollForTask()
     * - .putPipelineDefinition()
     * - .validatePipelineDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifWorkerGroup(value, operator) {
        return this.if(`datapipeline:workerGroup`, value, operator || 'ArnEquals');
    }
}
exports.Datapipeline = Datapipeline;
//# sourceMappingURL=data:application/json;base64,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