"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithResources = void 0;
const _3_actions_1 = require("./3-actions");
/**
 * Adds "resource" functionality to the Policy Statement
 */
class PolicyStatementWithResources extends _3_actions_1.PolicyStatementWithActions {
    constructor() {
        super(...arguments);
        this.useNotResources = false;
        this.resources = [];
        this.cdkResourcesApplied = false;
    }
    /**
     * Injects resources into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyResources();
            return super.toJSON();
        }
        const mode = this.useNotResources ? 'NotResource' : 'Resource';
        const statement = super.toJSON();
        const self = this;
        this.ensureResource();
        statement[mode] = this.resources.filter((elem, pos) => {
            return self.resources.indexOf(elem) == pos;
        });
        return statement;
    }
    toStatementJson() {
        this.ensureResource();
        this.cdkApplyResources();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyResources() {
        if (!this.cdkResourcesApplied) {
            const mode = this.useNotResources ? 'addNotResources' : 'addResources';
            const self = this;
            const uniqueResources = this.resources.filter((elem, pos) => {
                return self.resources.indexOf(elem) == pos;
            });
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueResources);
            this.cdkResourcesApplied = true;
        }
    }
    /**
     * Switches the statement to use [`NotResource`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html).
     */
    notResources() {
        this.useNotResources = true;
        return this;
    }
    /**
     * Checks weather any resource was applied to the policy.
     */
    hasResources() {
        return this.resources.length > 0;
    }
    /**
     * Limit statement to specified resources.
     *
     * To allow all resources, pass `*`
     */
    on(...arns) {
        this.resources.push(...arns);
        return this;
    }
    ensureResource() {
        if (this.hasResources())
            return;
        // @ts-ignore only available after swapping 1-base
        if (this.hasResource)
            return;
        // @ts-ignore only available after swapping 1-base
        if (this.hasPrincipal)
            return; //assume policies may not have resources
        // a statement requires resources. if none was added, we assume the user wants all resources
        this.resources.push('*');
    }
    /**
     * Dummy method. Will be overridden by 6-principal.ts
     *
     * We just need it here so we can reference it in method `ensureResource`
     */
    hasPrincipals() {
        return false;
    }
}
exports.PolicyStatementWithResources = PolicyStatementWithResources;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiNC1yZXNvdXJjZXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyI0LXJlc291cmNlcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw0Q0FBeUQ7QUFhekQ7O0dBRUc7QUFDSCxNQUFhLDRCQUE2QixTQUFRLHVDQUEwQjtJQUE1RTs7UUFDVSxvQkFBZSxHQUFHLEtBQUssQ0FBQztRQUN0QixjQUFTLEdBQWEsRUFBRSxDQUFDO1FBQzNCLHdCQUFtQixHQUFHLEtBQUssQ0FBQztJQTBGdEMsQ0FBQztJQXhGQzs7OztPQUlHO0lBQ0ksTUFBTTtRQUNYLGtEQUFrRDtRQUNsRCxJQUFJLE9BQU8sSUFBSSxDQUFDLFlBQVksSUFBSSxVQUFVLEVBQUU7WUFDMUMsSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUM7WUFDekIsT0FBTyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7U0FDdkI7UUFDRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztRQUMvRCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDakMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBRWxCLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztRQUV0QixTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsR0FBRyxFQUFFLEVBQUU7WUFDcEQsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUM7UUFDN0MsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRU0sZUFBZTtRQUNwQixJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7UUFDdEIsSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUM7UUFDekIsa0RBQWtEO1FBQ2xELE9BQU8sS0FBSyxDQUFDLGVBQWUsRUFBRSxDQUFDO0lBQ2pDLENBQUM7SUFFTyxpQkFBaUI7UUFDdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsRUFBRTtZQUM3QixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDO1lBQ3ZFLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztZQUNsQixNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsRUFBRTtnQkFDMUQsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUM7WUFDN0MsQ0FBQyxDQUFDLENBQUM7WUFDSCxrREFBa0Q7WUFDbEQsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsZUFBZSxDQUFDLENBQUM7WUFDL0IsSUFBSSxDQUFDLG1CQUFtQixHQUFHLElBQUksQ0FBQztTQUNqQztJQUNILENBQUM7SUFFRDs7T0FFRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksRUFBRSxDQUFDLEdBQUcsSUFBYztRQUN6QixJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVPLGNBQWM7UUFDcEIsSUFBSSxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQUUsT0FBTztRQUNoQyxrREFBa0Q7UUFDbEQsSUFBSSxJQUFJLENBQUMsV0FBVztZQUFFLE9BQU87UUFDN0Isa0RBQWtEO1FBQ2xELElBQUksSUFBSSxDQUFDLFlBQVk7WUFBRSxPQUFPLENBQUMsd0NBQXdDO1FBRXZFLDRGQUE0RjtRQUM1RixJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUMzQixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0NBQ0Y7QUE3RkQsb0VBNkZDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50V2l0aEFjdGlvbnMgfSBmcm9tICcuLzMtYWN0aW9ucyc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgUmVzb3VyY2VUeXBlcyB7XG4gIFtrZXk6IHN0cmluZ106IFJlc291cmNlVHlwZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBSZXNvdXJjZVR5cGUge1xuICBuYW1lOiBzdHJpbmc7XG4gIHVybDogc3RyaW5nO1xuICBhcm46IHN0cmluZztcbiAgY29uZGl0aW9uS2V5czogc3RyaW5nW107XG59XG5cbi8qKlxuICogQWRkcyBcInJlc291cmNlXCIgZnVuY3Rpb25hbGl0eSB0byB0aGUgUG9saWN5IFN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgUG9saWN5U3RhdGVtZW50V2l0aFJlc291cmNlcyBleHRlbmRzIFBvbGljeVN0YXRlbWVudFdpdGhBY3Rpb25zIHtcbiAgcHJpdmF0ZSB1c2VOb3RSZXNvdXJjZXMgPSBmYWxzZTtcbiAgcHJvdGVjdGVkIHJlc291cmNlczogc3RyaW5nW10gPSBbXTtcbiAgcHJpdmF0ZSBjZGtSZXNvdXJjZXNBcHBsaWVkID0gZmFsc2U7XG5cbiAgLyoqXG4gICAqIEluamVjdHMgcmVzb3VyY2VzIGludG8gdGhlIHN0YXRlbWVudC5cbiAgICpcbiAgICogT25seSByZWxldmFudCBmb3IgdGhlIG1haW4gcGFja2FnZS4gSW4gQ0RLIG1vZGUgdGhpcyBvbmx5IGNhbGxzIHN1cGVyLlxuICAgKi9cbiAgcHVibGljIHRvSlNPTigpOiBhbnkge1xuICAgIC8vIEB0cy1pZ25vcmUgb25seSBhdmFpbGFibGUgYWZ0ZXIgc3dhcHBpbmcgMS1iYXNlXG4gICAgaWYgKHR5cGVvZiB0aGlzLmFkZFJlc291cmNlcyA9PSAnZnVuY3Rpb24nKSB7XG4gICAgICB0aGlzLmNka0FwcGx5UmVzb3VyY2VzKCk7XG4gICAgICByZXR1cm4gc3VwZXIudG9KU09OKCk7XG4gICAgfVxuICAgIGNvbnN0IG1vZGUgPSB0aGlzLnVzZU5vdFJlc291cmNlcyA/ICdOb3RSZXNvdXJjZScgOiAnUmVzb3VyY2UnO1xuICAgIGNvbnN0IHN0YXRlbWVudCA9IHN1cGVyLnRvSlNPTigpO1xuICAgIGNvbnN0IHNlbGYgPSB0aGlzO1xuXG4gICAgdGhpcy5lbnN1cmVSZXNvdXJjZSgpO1xuXG4gICAgc3RhdGVtZW50W21vZGVdID0gdGhpcy5yZXNvdXJjZXMuZmlsdGVyKChlbGVtLCBwb3MpID0+IHtcbiAgICAgIHJldHVybiBzZWxmLnJlc291cmNlcy5pbmRleE9mKGVsZW0pID09IHBvcztcbiAgICB9KTtcblxuICAgIHJldHVybiBzdGF0ZW1lbnQ7XG4gIH1cblxuICBwdWJsaWMgdG9TdGF0ZW1lbnRKc29uKCk6IGFueSB7XG4gICAgdGhpcy5lbnN1cmVSZXNvdXJjZSgpO1xuICAgIHRoaXMuY2RrQXBwbHlSZXNvdXJjZXMoKTtcbiAgICAvLyBAdHMtaWdub3JlIG9ubHkgYXZhaWxhYmxlIGFmdGVyIHN3YXBwaW5nIDEtYmFzZVxuICAgIHJldHVybiBzdXBlci50b1N0YXRlbWVudEpzb24oKTtcbiAgfVxuXG4gIHByaXZhdGUgY2RrQXBwbHlSZXNvdXJjZXMoKSB7XG4gICAgaWYgKCF0aGlzLmNka1Jlc291cmNlc0FwcGxpZWQpIHtcbiAgICAgIGNvbnN0IG1vZGUgPSB0aGlzLnVzZU5vdFJlc291cmNlcyA/ICdhZGROb3RSZXNvdXJjZXMnIDogJ2FkZFJlc291cmNlcyc7XG4gICAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICAgIGNvbnN0IHVuaXF1ZVJlc291cmNlcyA9IHRoaXMucmVzb3VyY2VzLmZpbHRlcigoZWxlbSwgcG9zKSA9PiB7XG4gICAgICAgIHJldHVybiBzZWxmLnJlc291cmNlcy5pbmRleE9mKGVsZW0pID09IHBvcztcbiAgICAgIH0pO1xuICAgICAgLy8gQHRzLWlnbm9yZSBvbmx5IGF2YWlsYWJsZSBhZnRlciBzd2FwcGluZyAxLWJhc2VcbiAgICAgIHRoaXNbbW9kZV0oLi4udW5pcXVlUmVzb3VyY2VzKTtcbiAgICAgIHRoaXMuY2RrUmVzb3VyY2VzQXBwbGllZCA9IHRydWU7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFN3aXRjaGVzIHRoZSBzdGF0ZW1lbnQgdG8gdXNlIFtgTm90UmVzb3VyY2VgXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX25vdHJlc291cmNlLmh0bWwpLlxuICAgKi9cbiAgcHVibGljIG5vdFJlc291cmNlcygpIHtcbiAgICB0aGlzLnVzZU5vdFJlc291cmNlcyA9IHRydWU7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2tzIHdlYXRoZXIgYW55IHJlc291cmNlIHdhcyBhcHBsaWVkIHRvIHRoZSBwb2xpY3kuXG4gICAqL1xuICBwdWJsaWMgaGFzUmVzb3VyY2VzKCk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLnJlc291cmNlcy5sZW5ndGggPiAwO1xuICB9XG5cbiAgLyoqXG4gICAqIExpbWl0IHN0YXRlbWVudCB0byBzcGVjaWZpZWQgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBUbyBhbGxvdyBhbGwgcmVzb3VyY2VzLCBwYXNzIGAqYFxuICAgKi9cbiAgcHVibGljIG9uKC4uLmFybnM6IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5yZXNvdXJjZXMucHVzaCguLi5hcm5zKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHByaXZhdGUgZW5zdXJlUmVzb3VyY2UoKSB7XG4gICAgaWYgKHRoaXMuaGFzUmVzb3VyY2VzKCkpIHJldHVybjtcbiAgICAvLyBAdHMtaWdub3JlIG9ubHkgYXZhaWxhYmxlIGFmdGVyIHN3YXBwaW5nIDEtYmFzZVxuICAgIGlmICh0aGlzLmhhc1Jlc291cmNlKSByZXR1cm47XG4gICAgLy8gQHRzLWlnbm9yZSBvbmx5IGF2YWlsYWJsZSBhZnRlciBzd2FwcGluZyAxLWJhc2VcbiAgICBpZiAodGhpcy5oYXNQcmluY2lwYWwpIHJldHVybjsgLy9hc3N1bWUgcG9saWNpZXMgbWF5IG5vdCBoYXZlIHJlc291cmNlc1xuXG4gICAgLy8gYSBzdGF0ZW1lbnQgcmVxdWlyZXMgcmVzb3VyY2VzLiBpZiBub25lIHdhcyBhZGRlZCwgd2UgYXNzdW1lIHRoZSB1c2VyIHdhbnRzIGFsbCByZXNvdXJjZXNcbiAgICB0aGlzLnJlc291cmNlcy5wdXNoKCcqJyk7XG4gIH1cblxuICAvKipcbiAgICogRHVtbXkgbWV0aG9kLiBXaWxsIGJlIG92ZXJyaWRkZW4gYnkgNi1wcmluY2lwYWwudHNcbiAgICpcbiAgICogV2UganVzdCBuZWVkIGl0IGhlcmUgc28gd2UgY2FuIHJlZmVyZW5jZSBpdCBpbiBtZXRob2QgYGVuc3VyZVJlc291cmNlYFxuICAgKi9cbiAgcHVibGljIGhhc1ByaW5jaXBhbHMoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG59XG4iXX0=