# mautrix-facebook - A Matrix-Facebook Messenger puppeting bridge.
# Copyright (C) 2022 Tulir Asokan
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
from __future__ import annotations

from typing import TYPE_CHECKING, ClassVar

from asyncpg import Record
from attr import dataclass

from mautrix.util.async_db import Database

fake_db = Database.create("") if TYPE_CHECKING else None


@dataclass
class UserPortal:
    db: ClassVar[Database] = fake_db

    user: int
    portal: int
    portal_receiver: int

    @classmethod
    def _from_row(cls, row: Record | None) -> UserPortal | None:
        if row is None:
            return None
        return cls(**row)

    @classmethod
    async def all(cls, user: int) -> dict[int, UserPortal]:
        q = 'SELECT "user", portal, portal_receiver FROM user_portal WHERE "user"=$1'
        rows = await cls.db.fetch(q, user)
        return {up.portal: up for up in (cls._from_row(row) for row in rows)}

    @classmethod
    async def get(cls, user: int, portal: int, portal_receiver: int) -> UserPortal | None:
        q = (
            'SELECT "user", portal, portal_receiver FROM user_portal '
            'WHERE "user"=$1 AND portal=$2 AND portal_receiver=$3'
        )
        row = await cls.db.fetchrow(q, user, portal, portal_receiver)
        return cls._from_row(row)

    async def insert(self) -> None:
        q = 'INSERT INTO user_portal ("user", portal, portal_receiver) ' "VALUES ($1, $2, $3)"
        await self.db.execute(q, self.user, self.portal, self.portal_receiver)

    async def upsert(self) -> None:
        q = (
            'INSERT INTO user_portal ("user", portal, portal_receiver) '
            "VALUES ($1, $2, $3) "
            'ON CONFLICT ("user", portal, portal_receiver) DO NOTHING'
        )
        await self.db.execute(q, self.user, self.portal, self.portal_receiver)

    async def delete(self) -> None:
        q = 'DELETE FROM user_portal WHERE "user"=$1 AND portal=$2 AND portal_receiver=$3'
        await self.db.execute(q, self.user)

    @classmethod
    async def delete_all(cls, user: int) -> None:
        await cls.db.execute('DELETE FROM user_portal WHERE "user"=$1', user)
