from __future__ import print_function
from abc import ABCMeta, abstractmethod

import datetime

from algo_trade.event import FillEvent, OrderEvent
from tda.orders import equities


class ExecutionHandler(object):
    '''
    ExecutionHandler abstract class handles the interaction between a set of order objects generated by a Portfolio and the ultimate set of Fill objects that actually occur in the market.

    Handlers can be used to subclass simulated brokerages or live brokerages, with identical interfaces. Allows strategies to be backtested in a very similar manner to the live trading engine.
    '''
    __metaclass__ = ABCMeta

    @abstractmethod
    def execute_order(self, event):
        '''
        Takes OrderEvent object and executes it, producing an optional FillEvent that gets placed onto the Events queue.
        '''
        raise NotImplementedError('Should implement execute_order()')

class SimulatedExecutionHandler(ExecutionHandler):
    '''
    Simulated execution handler converts all OrderEvent objects into their equivalent FillEvent objects without latency, slippage, or fill-ratio issues.

    Allows straightforward "first go" test of any strategy, before implementation with a more sophisticated execution handler.

    :param events:  Queue of Event objects
    '''
    def __init__(self, events):
        self.events = events

    def execute_order(self, event):
        '''
        Converts OrderEvent into FillEvent objects naively
        '''
        if event.type == 'ORDER':
            fill_event = FillEvent(
                datetime.datetime.utcnow(), event.ticker, 'ARCA', event.quantity, event.direction, None
            )
            self.events.put(fill_event)

class TDAExecutionHandler(ExecutionHandler):
    '''
    TDAmeritrade execution handler, takes OrderEvent objects and submits trades via the TDA API. Current feature is HTTP trade submission only, does not return FillEvent information.

    Future - Research TDA API trade execution receipt

    :param client:  TDA API client that links handler to TDAmeritrade account
    :param acc_id:  TDA account number
    :param events:  Queue of Event objects
    '''
    def __init__(self, client, acc_id, events):
        self.client = client
        self.ACC_ID = acc_id
        self.events = events

    def _create_order(self, ticker, quantity, action, order_type, limit=None):
        '''
        Creates OrderEvent objects and adds them to the order_book set.

        :param ticker: Ticker symbol
        :param quantity: Number of shares to buy/sell
        :param action: BUY or SELL
        :param order_type: MARKET or LIMIT - currently only supports MARKET types
        '''
        return OrderEvent(
            ticker=ticker,
            quantity=quantity,
            action=action,
            order_type=order_type,
            limit=limit,
        )

    def _calculate_weight_change(self, target, current):
        '''
        Calculates the difference between tar_state and cur_state portfolios, returning how much each asset is under/over-weight

        :param tar_state: Pandas DataFrame of target asset weights in decimals
        :param cur_state: Pandas DataFrame of current asset weights in decimals
        '''
        diff_dict = dict()

        # Get difference from current holdings to target holdings
        for k, v in current.items():
            if k not in target:
                diff_dict[k] = -1 * v
            else:
                diff_dict[k] = target[k] - v

        # Check for new asset holdings in target
        for k, v in target.items():
            if k not in current:
                diff_dict[k] = v
        
        # Drop cash component
        diff_dict.pop('MMDA1', None)

        return diff_dict

    def _submitBuy(self, event):
        '''
        Takes OrderEvent and submits a BUY order to the TDA account
        '''
        if event.order_type == 'MARKET':
            self.client.place_order(
                self.ACC_ID,
                equities.equity_buy_market(
                    event.ticker,
                    event.quantity,
                ),
            )
        elif event.order_type == 'LIMIT' and event.limit:
            self.client.place_order(
                self.ACC_ID,
                equities.equity_buy_limit(
                    event.ticker,
                    event.quantity,
                    event.limit
                ),
            )
        else:
            raise Exception('Invalid BUY order.')

    def _submitSell(self, event):
        '''
        Takes OrderEvent and submits a SELL order to the TDA account
        '''
        if event.order_type == 'MARKET':
            self.client.place_order(
                self.ACC_ID,
                equities.equity_sell_market(
                    event.ticker,
                    event.quantity,
                ),
            )
        elif event.order_type == 'LIMIT' and event.limit:
            self.client.place_order(
                self.ACC_ID,
                equities.equity_sell_limit(
                    event.ticker,
                    event.quantity, 
                    event.limit,
                ),
            )
        else:
            raise Exception('Invalid SELL order.')

    def rebalance(self, balance, price, target, current = None):
        '''
        Returns set of OrderEvents created from current TDA account balances and target asset weights. If cur_df provided, will generate BUY and SELL orders accordingly to rebalance portfolio.

        :param balance: Float value of current account balance
        :param price: Dictionary of relevant ticker prices - should include all unique tickers present in target and current
        :param tar_df: Dictionary of target asset weights in float
        :param cur_df: Dictionary of current asset weights in float
        '''
        order_book = set()

        if current is not None:
            target = self._calculate_weight_change(target, current)

        for k, v in target.items():
            quantity = int(v * balance / price[k])

            print(type(k), k)

            if k == 'MMDA1':
                continue

            if quantity > 0:
                order_book.add( self._create_order(k, quantity, 'BUY', 'MARKET') )
            elif quantity < 0:
                order_book.add( self._create_order(k, abs(quantity), 'SELL', 'MARKET') )

        return order_book

    def execute_order(self, event):
        '''
        Virtual method - takes OrderEvent and calls self._submitBuy() and self._submitSell() in accordance to the order type
        '''
        if event.action == 'BUY':
            self._submitBuy(event)

        if event.action =='SELL':
            self._submitSell(event)