"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mq = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const shared_1 = require("../shared");
/**
 * Statement provider for service [mq](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmq.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mq extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mq](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmq.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mq';
        this.accessLevelList = {
            Write: [
                'CreateBroker',
                'CreateConfiguration',
                'CreateUser',
                'DeleteBroker',
                'DeleteUser',
                'RebootBroker',
                'UpdateBroker',
                'UpdateConfiguration',
                'UpdateUser'
            ],
            Tagging: [
                'CreateTags',
                'DeleteTags'
            ],
            Read: [
                'DescribeBroker',
                'DescribeBrokerEngineTypes',
                'DescribeBrokerInstanceOptions',
                'DescribeConfiguration',
                'DescribeConfigurationRevision',
                'DescribeUser'
            ],
            List: [
                'ListBrokers',
                'ListConfigurationRevisions',
                'ListConfigurations',
                'ListTags',
                'ListUsers'
            ]
        };
    }
    /**
     * Grants permission to create a broker
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:CreateSecurityGroup
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeInternetGateways
     * - ec2:DescribeNetworkInterfacePermissions
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - ec2:ModifyNetworkInterfaceAttribute
     * - iam:CreateServiceLinkedRole
     * - route53:AssociateVPCWithHostedZone
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-post
     */
    toCreateBroker() {
        return this.to('CreateBroker');
    }
    /**
     * Grants permission to create a new configuration for the specified configuration name. Amazon MQ uses the default configuration (the engine type and engine version)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-post
     */
    toCreateConfiguration() {
        return this.to('CreateConfiguration');
    }
    /**
     * Grants permission to create tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-post
     */
    toCreateTags() {
        return this.to('CreateTags');
    }
    /**
     * Grants permission to create an ActiveMQ user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-post
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete a broker
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteNetworkInterfacePermission
     * - ec2:DeleteVpcEndpoints
     * - ec2:DetachNetworkInterface
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-delete
     */
    toDeleteBroker() {
        return this.to('DeleteBroker');
    }
    /**
     * Grants permission to delete tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-delete
     */
    toDeleteTags() {
        return this.to('DeleteTags');
    }
    /**
     * Grants permission to delete an ActiveMQ user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-delete
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to return information about the specified broker
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-get
     */
    toDescribeBroker() {
        return this.to('DescribeBroker');
    }
    /**
     * Grants permission to return information about broker engines
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/broker-engine-types.html#broker-engine-types-http-methods
     */
    toDescribeBrokerEngineTypes() {
        return this.to('DescribeBrokerEngineTypes');
    }
    /**
     * Grants permission to return information about the broker instance options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/broker-instance-options.html#broker-engine-types-http-methods
     */
    toDescribeBrokerInstanceOptions() {
        return this.to('DescribeBrokerInstanceOptions');
    }
    /**
     * Grants permission to return information about the specified configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration.html#rest-api-configuration-methods-get
     */
    toDescribeConfiguration() {
        return this.to('DescribeConfiguration');
    }
    /**
     * Grants permission to return the specified configuration revision for the specified configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration-revision.html#rest-api-configuration-revision-methods-get
     */
    toDescribeConfigurationRevision() {
        return this.to('DescribeConfigurationRevision');
    }
    /**
     * Grants permission to return information about an ActiveMQ user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-get
     */
    toDescribeUser() {
        return this.to('DescribeUser');
    }
    /**
     * Grants permission to return a list of all brokers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-get
     */
    toListBrokers() {
        return this.to('ListBrokers');
    }
    /**
     * Grants permission to return a list of all existing revisions for the specified configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-revisions.html#rest-api-revisions-methods-get
     */
    toListConfigurationRevisions() {
        return this.to('ListConfigurationRevisions');
    }
    /**
     * Grants permission to return a list of all configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-get
     */
    toListConfigurations() {
        return this.to('ListConfigurations');
    }
    /**
     * Grants permission to return a list of tags
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-get
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to return a list of all ActiveMQ users
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-users.html#rest-api-users-methods-get
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to reboot a broker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-restart.html#rest-api-reboot-methods-post
     */
    toRebootBroker() {
        return this.to('RebootBroker');
    }
    /**
     * Grants permission to add a pending configuration change to a broker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-get
     */
    toUpdateBroker() {
        return this.to('UpdateBroker');
    }
    /**
     * Grants permission to update the specified configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration.html#rest-api-configuration-methods-put
     */
    toUpdateConfiguration() {
        return this.to('UpdateConfiguration');
    }
    /**
     * Grants permission to update the information for an ActiveMQ user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-put
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Adds a resource of type brokers to the statement
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/amazon-mq-how-it-works.html
     *
     * @param brokerId - Identifier for the brokerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBrokers(brokerId, account, region, partition) {
        return this.on(`arn:${partition || Mq.defaultPartition}:mq:${region || '*'}:${account || '*'}:broker:${brokerId}`);
    }
    /**
     * Adds a resource of type configurations to the statement
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/amazon-mq-how-it-works.html
     *
     * @param configurationId - Identifier for the configurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurations(configurationId, account, region, partition) {
        return this.on(`arn:${partition || Mq.defaultPartition}:mq:${region || '*'}:${account || '*'}:configuration:${configurationId}`);
    }
}
exports.Mq = Mq;
_a = JSII_RTTI_SYMBOL_1;
Mq[_a] = { fqn: "iam-floyd.Mq", version: "0.369.0" };
//# sourceMappingURL=data:application/json;base64,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