"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithPrincipal = exports.PrincipalType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _5_effect_1 = require("./5-effect");
var PrincipalType;
(function (PrincipalType) {
    PrincipalType["AWS"] = "AWS";
    PrincipalType["FEDERATED"] = "Federated";
    PrincipalType["CANONICAL_USER"] = "CanonicalUser";
    PrincipalType["SERVICE"] = "Service";
})(PrincipalType = exports.PrincipalType || (exports.PrincipalType = {}));
/**
 * Adds "principal" functionality to the Policy Statement
 */
class PolicyStatementWithPrincipal extends _5_effect_1.PolicyStatementWithEffect {
    constructor() {
        super(...arguments);
        this.useNotPrincipals = false;
        this.myPrincipals = {};
    }
    /**
     * Injects principals into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyPrincipals();
            return super.toJSON();
        }
        const mode = this.useNotPrincipals ? 'NotPrincipal' : 'Principal';
        const statement = super.toJSON();
        if (this.hasPrincipals()) {
            statement[mode] = this.myPrincipals;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyPrincipals();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyPrincipals() { }
    /**
     * Switches the statement to use [`notPrincipal`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html).
     */
    notPrincipals() {
        this.useNotPrincipals = true;
        return this;
    }
    /**
     * Checks weather a principal was applied to the policy
     */
    hasPrincipals() {
        return Object.keys(this.myPrincipals).length > 0;
    }
    /**
     * Adds a principal to the statement
     *
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service**
     * @param principal The principal string
     */
    addPrincipal(prefix, principal) {
        this.skipAutoResource = true;
        if (!(prefix in this.myPrincipals)) {
            this.myPrincipals[prefix] = [];
        }
        this.myPrincipals[prefix].push(principal);
        return this;
    }
    /**
     * Adds any principal to the statement
     *
     * @param arn The ARN of the principal
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service** - Default: **AWS**
     */
    for(arn, prefix) {
        return this.addPrincipal(prefix || PrincipalType.AWS, arn);
    }
    /**
     * Adds one or more account principals to the statement
     *
     * @param accounts ID of the AWS account
     */
    forAccount(...accounts) {
        accounts.forEach((account) => this.addPrincipal(PrincipalType.AWS, `arn:${PolicyStatementWithPrincipal.defaultPartition}:iam::${account}:root`));
        return this;
    }
    /**
     * Adds one or more [federated](https://aws.amazon.com/identity/federation/) (web identity) principals to the statement
     *
     * @param providers ID of the AWS account
     */
    forFederated(...providers) {
        providers.forEach((provider) => this.addPrincipal(PrincipalType.FEDERATED, provider));
        return this;
    }
    /**
     * Adds a federated [AWS Cognito](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc_cognito.html) principal to the statement
     */
    forFederatedCognito() {
        return this.forFederated('cognito-identity.amazonaws.com');
    }
    /**
     * Adds a federated [Amazon](https://login.amazon.com/) principal to the statement
     */
    forFederatedAmazon() {
        return this.forFederated('www.amazon.com');
    }
    /**
     * Adds a federated [Facebook](https://developers.facebook.com/docs/facebook-login) principal to the statement
     */
    forFederatedFacebook() {
        return this.forFederated('graph.facebook.com');
    }
    /**
     * Adds a federated [Google](https://developers.google.com/identity/protocols/oauth2/openid-connect) principal to the statement
     */
    forFederatedGoogle() {
        return this.forFederated('accounts.google.com');
    }
    /**
     * Adds one or more canonical user principals to the statement
     *
     * @param userIDs The user ID
     *
     * You can [find the canonical user ID](https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html#FindingCanonicalId) for your AWS account in the AWS Management Console. The canonical user ID for an AWS account is specific to the account. You can retrieve the canonical user ID for your AWS account as either the root user or an IAM user.
     */
    forCanonicalUser(...userIDs) {
        userIDs.forEach((userID) => this.addPrincipal(PrincipalType.CANONICAL_USER, userID));
        return this;
    }
    /**
     * Adds one or more federated SAML principals to the statement
     *
     * @param account ID of the AWS account
     * @param providerNames Name of the SAML provider
     */
    forSaml(account, ...providerNames) {
        providerNames.forEach((providerName) => this.forFederated(`arn:${PolicyStatementWithPrincipal.defaultPartition}:iam::${account}:saml-provider/${providerName}`));
        return this;
    }
    /**
     * Adds one or more IAM user principals to the statement
     *
     * @param account ID of the AWS account
     * @param users Name of the IAM user
     */
    forUser(account, ...users) {
        users.forEach((user) => this.addPrincipal(PrincipalType.AWS, `arn:${PolicyStatementWithPrincipal.defaultPartition}:iam::${account}:user/${user}`));
        return this;
    }
    /**
     * Adds one or more IAM role principals to the statement
     *
     * @param account ID of the AWS account
     * @param roles Name of the IAM role
     */
    forRole(account, ...roles) {
        roles.forEach((role) => this.addPrincipal(PrincipalType.AWS, `arn:${PolicyStatementWithPrincipal.defaultPartition}:iam::${account}:role/${role}`));
        return this;
    }
    /**
     * Adds one or more specific assumed role session principals to the statement
     *
     * @param account ID of the AWS account
     * @param roleName Name of the IAM role
     * @param sessionNames Name of the session. You cannot use a wildcard (`*`) to mean *all sessions*. Principals must always name a specific session
     */
    forAssumedRoleSession(account, roleName, ...sessionNames) {
        sessionNames.forEach((sessionName) => {
            this.addPrincipal(PrincipalType.AWS, `arn:${PolicyStatementWithPrincipal.defaultPartition}:sts::${account}:assumed-role/${roleName}/${sessionName}`);
        });
        return this;
    }
    /**
     * Adds one or more service principals to the statement
     *
     * @param services Long version of the service name. Usually in the format: `long_service-name.amazonaws.com`
     *
     * The service principal is defined by the service. To learn the service principal for a service, see the documentation for that service. For some services, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html) and look for the services that have **Yes** in the **Service-Linked Role** column. Choose a **Yes** with a link to view the service-linked role documentation for that service. View the **Service-Linked Role Permissions** section for that service to view the service principal.
     */
    forService(...services) {
        services.forEach((service) => this.addPrincipal(PrincipalType.SERVICE, service));
        return this;
    }
    /**
     * Grants public asses
     *
     * **EVERYONE IN THE WORLD HAS ACCESS**
     *
     * We strongly recommend that you do not use a wildcard in the Principal element in a role's trust policy unless you otherwise restrict access through a Condition element in the policy. Otherwise, any IAM user in any account in your partition can access the role.
     */
    forPublic() {
        return this.addPrincipal(PrincipalType.AWS, '*');
    }
}
exports.PolicyStatementWithPrincipal = PolicyStatementWithPrincipal;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithPrincipal[_a] = { fqn: "iam-floyd.PolicyStatementWithPrincipal", version: "0.369.0" };
//# sourceMappingURL=data:application/json;base64,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