"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerService = exports.Database = exports.KeyCloak = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const certmgr = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const rds = require("@aws-cdk/aws-rds");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
// regional availibility for aurora serverless
// see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.AuroraFeaturesRegionsDBEngines.grids.html
const AURORA_SERVERLESS_SUPPORTED_REGIONS = [
    'us-east-1',
    'us-east-2',
    'us-west-1',
    'us-west-2',
    'ap-south-1',
    'ap-northeast-1',
    'ap-northeast-2',
    'ap-southeast-1',
    'ap-southeast-2',
    'ca-central-1',
    'eu-central-1',
    'eu-west-1',
    'eu-west-2',
    'eu-west-3',
    'cn-northwest-1',
];
const KEYCLOAK_VERSION = '12.0.4';
const KEYCLOAK_DOCKER_IMAGE_URI_MAP = {
    'aws': `jboss/keycloak:${KEYCLOAK_VERSION}`,
    'aws-cn': `048912060910.dkr.ecr.cn-northwest-1.amazonaws.com.cn/dockerhub/jboss/keycloak:${KEYCLOAK_VERSION}`,
};
/**
 * @stability stable
 */
class KeyCloak extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d;
        super(scope, id);
        const region = cdk.Stack.of(this).region;
        const regionIsResolved = !cdk.Token.isUnresolved(region);
        if (props.auroraServerless && regionIsResolved && !AURORA_SERVERLESS_SUPPORTED_REGIONS.includes(region)) {
            throw new Error(`Aurora serverless is not supported in ${region}`);
        }
        this.vpc = (_d = props.vpc) !== null && _d !== void 0 ? _d : getOrCreateVpc(this);
        this.db = this.addDatabase({
            vpc: this.vpc,
            databaseSubnets: props.databaseSubnets,
            instanceType: props.databaseInstanceType,
            instanceEngine: props.instanceEngine,
            clusterEngine: props.clusterEngine,
            auroraServerless: props.auroraServerless,
            singleDbInstance: props.singleDbInstance,
            backupRetention: props.backupRetention,
        });
        this.addKeyCloakContainerService({
            database: this.db,
            vpc: this.vpc,
            publicSubnets: props.publicSubnets,
            privateSubnets: props.privateSubnets,
            keycloakSecret: this._generateKeycloakSecret(),
            certificate: certmgr.Certificate.fromCertificateArn(this, 'ACMCert', props.certificateArn),
            bastion: props.bastion,
            nodeCount: props.nodeCount,
            stickinessCookieDuration: props.stickinessCookieDuration,
            autoScaleTask: props.autoScaleTask,
            env: props.env,
        });
        if (!cdk.Stack.of(this).templateOptions.description) {
            cdk.Stack.of(this).templateOptions.description = '(SO8021) - Deploy keycloak on AWS with cdk-keycloak construct library';
        }
    }
    /**
     * @stability stable
     */
    addDatabase(props) {
        return new Database(this, 'Database', props);
    }
    /**
     * @stability stable
     */
    addKeyCloakContainerService(props) {
        return new ContainerService(this, 'KeyCloakContainerSerivce', props);
    }
    _generateKeycloakSecret() {
        return new secretsmanager.Secret(this, 'KCSecret', {
            generateSecretString: {
                generateStringKey: 'password',
                excludePunctuation: true,
                passwordLength: 12,
                secretStringTemplate: JSON.stringify({ username: 'keycloak' }),
            },
        });
    }
}
exports.KeyCloak = KeyCloak;
_a = JSII_RTTI_SYMBOL_1;
KeyCloak[_a] = { fqn: "cdk-keycloak.KeyCloak", version: "0.2.21" };
/**
 * Represents the database instance or database cluster.
 *
 * @stability stable
 */
class Database extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this._mysqlListenerPort = 3306;
        this.vpc = props.vpc;
        const config = props.auroraServerless ? this._createServerlessCluster(props)
            : props.singleDbInstance ? this._createRdsInstance(props) : this._createRdsCluster(props);
        this.secret = config.secret;
        // allow internally from the same security group
        config.connections.allowInternally(ec2.Port.tcp(this._mysqlListenerPort));
        // allow from the whole vpc cidr
        config.connections.allowFrom(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(this._mysqlListenerPort));
        this.clusterEndpointHostname = config.endpoint;
        this.clusterIdentifier = config.identifier;
        this.connections = config.connections;
        printOutput(this, 'DBSecretArn', config.secret.secretArn);
        printOutput(this, 'clusterEndpointHostname', this.clusterEndpointHostname);
        printOutput(this, 'clusterIdentifier', this.clusterIdentifier);
    }
    _createRdsInstance(props) {
        var _d, _e, _f;
        const dbInstance = new rds.DatabaseInstance(this, 'DBInstance', {
            vpc: props.vpc,
            databaseName: 'keycloak',
            vpcSubnets: props.databaseSubnets,
            engine: (_d = props.instanceEngine) !== null && _d !== void 0 ? _d : rds.DatabaseInstanceEngine.mysql({
                version: rds.MysqlEngineVersion.VER_8_0_21,
            }),
            storageEncrypted: true,
            backupRetention: (_e = props.backupRetention) !== null && _e !== void 0 ? _e : cdk.Duration.days(7),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceType: (_f = props.instanceType) !== null && _f !== void 0 ? _f : new ec2.InstanceType('r5.large'),
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.mysql8.0'),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        return {
            connections: dbInstance.connections,
            endpoint: dbInstance.dbInstanceEndpointAddress,
            identifier: dbInstance.instanceIdentifier,
            secret: dbInstance.secret,
        };
    }
    // create a RDS for MySQL DB cluster
    _createRdsCluster(props) {
        var _d, _e, _f;
        const dbCluster = new rds.DatabaseCluster(this, 'DBCluster', {
            engine: (_d = props.clusterEngine) !== null && _d !== void 0 ? _d : rds.DatabaseClusterEngine.auroraMysql({
                version: rds.AuroraMysqlEngineVersion.VER_2_09_1,
            }),
            defaultDatabaseName: 'keycloak',
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceProps: {
                vpc: props.vpc,
                vpcSubnets: props.databaseSubnets,
                instanceType: (_e = props.instanceType) !== null && _e !== void 0 ? _e : new ec2.InstanceType('r5.large'),
            },
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
            backup: {
                retention: (_f = props.backupRetention) !== null && _f !== void 0 ? _f : cdk.Duration.days(7),
            },
            storageEncrypted: true,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
    _createServerlessCluster(props) {
        var _d;
        const dbCluster = new rds.ServerlessCluster(this, 'AuroraServerlessCluster', {
            engine: rds.DatabaseClusterEngine.AURORA_MYSQL,
            vpc: props.vpc,
            defaultDatabaseName: 'keycloak',
            vpcSubnets: props.databaseSubnets,
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            backupRetention: (_d = props.backupRetention) !== null && _d !== void 0 ? _d : cdk.Duration.days(7),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
}
exports.Database = Database;
_b = JSII_RTTI_SYMBOL_1;
Database[_b] = { fqn: "cdk-keycloak.Database", version: "0.2.21" };
/**
 * @stability stable
 */
class ContainerService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        const vpc = props.vpc;
        const cluster = new ecs.Cluster(this, 'Cluster', { vpc });
        const taskRole = new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('ecs.amazonaws.com'), new iam.ServicePrincipal('ecs-tasks.amazonaws.com')),
        });
        const taskDefinition = new ecs.FargateTaskDefinition(this, 'TaskDef', {
            cpu: 4096,
            memoryLimitMiB: 30720,
            executionRole: taskRole,
        });
        const logGroup = new logs.LogGroup(this, 'LogGroup', {
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const kc = taskDefinition.addContainer('keycloak', {
            image: ecs.ContainerImage.fromRegistry(this.getKeyCloakDockerImageUri()),
            environment: Object.assign({
                DB_ADDR: props.database.clusterEndpointHostname,
                DB_DATABASE: 'keycloak',
                DB_PORT: '3306',
                DB_USER: 'admin',
                DB_VENDOR: 'mysql',
                // KEYCLOAK_LOGLEVEL: 'DEBUG',
                JDBC_PARAMS: 'useSSL=false',
                JGROUPS_DISCOVERY_PROTOCOL: 'JDBC_PING',
            }, props.env),
            secrets: {
                DB_PASSWORD: ecs.Secret.fromSecretsManager(props.database.secret, 'password'),
                KEYCLOAK_USER: ecs.Secret.fromSecretsManager(props.keycloakSecret, 'username'),
                KEYCLOAK_PASSWORD: ecs.Secret.fromSecretsManager(props.keycloakSecret, 'password'),
            },
            logging: ecs.LogDrivers.awsLogs({
                streamPrefix: 'keycloak',
                logGroup,
            }),
        });
        kc.addPortMappings({ containerPort: 8443 }, // HTTPS web port
        { containerPort: 7600 }, // jgroups-tcp
        { containerPort: 57600 }, // jgroups-tcp-fd
        { containerPort: 55200, protocol: ecs.Protocol.UDP }, // jgroups-udp
        { containerPort: 54200, protocol: ecs.Protocol.UDP });
        // we need extra privileges to fetch keycloak docker images from China mirror site
        (_d = taskDefinition.executionRole) === null || _d === void 0 ? void 0 : _d.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        this.service = new ecs.FargateService(this, 'Service', {
            cluster,
            taskDefinition,
            circuitBreaker: props.circuitBreaker ? { rollback: true } : undefined,
            desiredCount: (_e = props.nodeCount) !== null && _e !== void 0 ? _e : 2,
            healthCheckGracePeriod: cdk.Duration.seconds(120),
        });
        // we need to allow traffic from the same secret group for keycloak cluster with jdbc_ping
        this.service.connections.allowFrom(this.service.connections, ec2.Port.tcp(7600), 'kc jgroups-tcp');
        this.service.connections.allowFrom(this.service.connections, ec2.Port.tcp(57600), 'kc jgroups-tcp-fd');
        this.service.connections.allowFrom(this.service.connections, ec2.Port.udp(55200), 'kc jgroups-udp');
        this.service.connections.allowFrom(this.service.connections, ec2.Port.udp(54200), 'kc jgroups-udp-fd');
        if (props.autoScaleTask) {
            const minCapacity = (_g = (_f = props.autoScaleTask.min) !== null && _f !== void 0 ? _f : props.nodeCount) !== null && _g !== void 0 ? _g : 2;
            const scaling = this.service.autoScaleTaskCount({
                minCapacity,
                maxCapacity: (_h = props.autoScaleTask.max) !== null && _h !== void 0 ? _h : minCapacity + 5,
            });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_j = props.autoScaleTask.targetCpuUtilization) !== null && _j !== void 0 ? _j : 75,
            });
        }
        ;
        const alb = new elbv2.ApplicationLoadBalancer(this, 'ALB', {
            vpc,
            vpcSubnets: props.publicSubnets,
            internetFacing: true,
        });
        printOutput(this, 'EndpointURL', `https://${alb.loadBalancerDnsName}`);
        const listener = alb.addListener('HttpsListener', {
            protocol: elbv2.ApplicationProtocol.HTTPS,
            certificates: [{ certificateArn: props.certificate.certificateArn }],
        });
        listener.addTargets('ECSTarget', {
            targets: [this.service],
            // set slow_start.duration_seconds to 60
            // see https://docs.aws.amazon.com/cli/latest/reference/elbv2/modify-target-group-attributes.html
            slowStart: cdk.Duration.seconds(60),
            stickinessCookieDuration: (_k = props.stickinessCookieDuration) !== null && _k !== void 0 ? _k : cdk.Duration.days(1),
            port: 8443,
            protocol: elbv2.ApplicationProtocol.HTTPS,
        });
        // allow task execution role to read the secrets
        props.database.secret.grantRead(taskDefinition.executionRole);
        props.keycloakSecret.grantRead(taskDefinition.executionRole);
        // allow ecs task connect to database
        props.database.connections.allowDefaultPortFrom(this.service);
        // create a bastion host
        if (props.bastion === true) {
            const bast = new ec2.BastionHostLinux(this, 'Bast', {
                vpc,
                instanceType: new ec2.InstanceType('m5.large'),
            });
            props.database.connections.allowDefaultPortFrom(bast);
        }
    }
    getImageUriFromMap(map, id) {
        const stack = cdk.Stack.of(this);
        if (cdk.Token.isUnresolved(stack.region)) {
            const mapping = {};
            for (const [partition, uri] of Object.entries(map)) {
                mapping[partition] = { uri };
            }
            const imageMap = new cdk.CfnMapping(this, id, { mapping });
            return imageMap.findInMap(cdk.Aws.PARTITION, 'uri');
        }
        else {
            if (stack.region.startsWith('cn-')) {
                return map['aws-cn'];
            }
            else {
                return map.aws;
            }
        }
    }
    getKeyCloakDockerImageUri() {
        return this.getImageUriFromMap(KEYCLOAK_DOCKER_IMAGE_URI_MAP, 'KeycloakImageMap');
    }
}
exports.ContainerService = ContainerService;
_c = JSII_RTTI_SYMBOL_1;
ContainerService[_c] = { fqn: "cdk-keycloak.ContainerService", version: "0.2.21" };
/**
 * Create or import VPC
 * @param scope the cdk scope
 */
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1' ?
        ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
function printOutput(scope, id, key) {
    new cdk.CfnOutput(scope, id, { value: String(key) });
}
//# sourceMappingURL=data:application/json;base64,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