"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the CodeCommit Action detect changes.
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    CodeCommitTrigger["NONE"] = "None";
    /**
     * CodePipeline will poll the repository to detect changes.
     */
    CodeCommitTrigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * This is the default method of detecting changes.
     */
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 */
class CodeCommitSourceAction extends action_1.Action {
    constructor(props) {
        const branch = props.branch || 'master';
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    bound(_scope, stage, options) {
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            this.props.repository.onCommit(stage.pipeline.node.uniqueId + 'EventRule', {
                target: new targets.CodePipeline(stage.pipeline),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
            },
        };
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
//# sourceMappingURL=data:application/json;base64,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