"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * CodePipeline Action to deploy an ECS Service.
 */
class EcsDeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'ECS',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [determineInputArtifact(props)],
            resource: props.service
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // permissions based on CodePipeline documentation:
        // https://docs.aws.amazon.com/codepipeline/latest/userguide/how-to-custom-role.html#how-to-update-role-new-services
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'ecs:DescribeTaskDefinition',
                'ecs:DescribeTasks',
                'ecs:ListTasks',
                'ecs:RegisterTaskDefinition',
                'ecs:UpdateService',
            ],
            resources: ['*']
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ec2.amazonaws.com',
                        'ecs-tasks.amazonaws.com',
                    ],
                }
            }
        }));
        options.bucket.grantRead(options.role);
        return {
            configuration: {
                ClusterName: this.props.service.cluster.clusterName,
                ServiceName: this.props.service.serviceName,
                FileName: this.props.imageFile && this.props.imageFile.fileName,
            },
        };
    }
}
exports.EcsDeployAction = EcsDeployAction;
function determineInputArtifact(props) {
    if (props.imageFile && props.input) {
        throw new Error("Exactly one of 'input' or 'imageFile' can be provided in the ECS deploy Action");
    }
    if (props.imageFile) {
        return props.imageFile.artifact;
    }
    if (props.input) {
        return props.input;
    }
    throw new Error("Specifying one of 'input' or 'imageFile' is required for the ECS deploy Action");
}
//# sourceMappingURL=data:application/json;base64,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