""" Functions for loading experimental spectra from mzml files.
"""
import os

import numpy as np
import pandas as pd
from pyteomics import mzml

from inspire.constants import (
    CHARGE_KEY,
    INTENSITIES_KEY,
    MZS_KEY,
    RT_KEY,
    SCAN_KEY,
    SOURCE_KEY,
)


def process_mzml_file(mzml_filename, scan_ids, with_charge=False, with_retention_time=False):
    """ Function to process an MzML file to find matches with scan IDs.

    Parameters
    ----------
    mzml_filename : str
        The mzml file from which we are reading.
    scan_ids : list of int
        A list of the scan IDs we require.

    Returns
    -------
    scans_df : pd.DataFrame
        A DataFrame of scan results.
    """
    ion_list = []
    intensities_list = []
    scan_id_list = []
    mzml_filenames = []
    filename = mzml_filename.split('/')[-1]

    if os.path.exists(mzml_filename.replace('.mzML', '_calibrated.mzML')):
        input_name = mzml_filename.replace('.mzML', '_calibrated.mzML')
    elif os.path.exists(mzml_filename):
        input_name = mzml_filename
    else:
        input_name = mzml_filename.replace('.mzML', '_uncalibrated.mzML')

    if with_charge:
        charge_list = []

    if with_retention_time:
        rt_list = []

    with mzml.read(input_name) as reader:
        for spectrum in reader:
            scan_id = int(spectrum['id'].split('scan=')[1])
            if scan_ids is None or scan_id in scan_ids:
                mzml_filenames.append(filename)
                scan_id_list.append(scan_id)
                intensities_list.append(np.array(list(spectrum['intensity array'])))
                ion_list.append(np.array(list(spectrum['m/z array'])))

                if with_charge:
                    charge_list.append(
                        int(spectrum['precursorList']['precursor'][0]['selectedIonList'][
                            'selectedIon'
                        ][0]['charge state'])
                    )

                if with_retention_time:
                    rt_list.append(float(spectrum['scanList']['scan'][0]['scan start time']))

    scans_df =  pd.DataFrame(
        {
            SOURCE_KEY: pd.Series(mzml_filenames),
            SCAN_KEY: pd.Series(scan_id_list),
            INTENSITIES_KEY: pd.Series(intensities_list),
            MZS_KEY: pd.Series(ion_list)
        }
    )
    if with_charge:
        scans_df[CHARGE_KEY] = pd.Series(charge_list)
    if with_retention_time:
        scans_df[RT_KEY] = pd.Series(rt_list)

    scans_df = scans_df.drop_duplicates(subset=[SOURCE_KEY, SCAN_KEY])

    return scans_df
