"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
/**
 * The content type for a fixed response
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * Define a new listener rule
 */
class ApplicationListenerRule extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        this.actions = [];
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (!props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.anyValue({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.anyValue({ produce: () => this.actions }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * Add a non-standard condition to this rule
     */
    setCondition(field, values) {
        this.conditions[field] = values;
    }
    /**
     * Add a TargetGroup to load balance to
     */
    addTargetGroup(targetGroup) {
        this.actions.push({
            targetGroupArn: targetGroup.targetGroupArn,
            type: 'forward',
        });
        targetGroup.registerListener(this.listener, this);
    }
    /**
     * Add a fixed response
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.actions.push({
            fixedResponseConfig: fixedResponse,
            type: 'fixed-response',
        });
    }
    /**
     * Add a redirect response
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.actions.push({
            redirectConfig: redirectResponse,
            type: 'redirect',
        });
    }
    /**
     * Validate the rule
     */
    validate() {
        if (this.actions.length === 0) {
            return ['Listener rule needs at least one action'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const ret = new Array();
        for (const [field, values] of Object.entries(this.conditions)) {
            if (values !== undefined) {
                ret.push({ field, values });
            }
        }
        return ret;
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,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