import { Construct, Duration, IResource } from '@aws-cdk/core';
import { BaseListener } from '../shared/base-listener';
import { HealthCheck } from '../shared/base-target-group';
import { Protocol, SslPolicy } from '../shared/enums';
import { IListenerCertificate } from '../shared/listener-certificate';
import { INetworkLoadBalancer } from './network-load-balancer';
import { INetworkLoadBalancerTarget, INetworkTargetGroup, NetworkTargetGroup } from './network-target-group';
/**
 * Basic properties for a Network Listener
 */
export interface BaseNetworkListenerProps {
    /**
     * The port on which the listener listens for requests.
     */
    readonly port: number;
    /**
     * Default target groups to load balance to
     *
     * @default - None.
     */
    readonly defaultTargetGroups?: INetworkTargetGroup[];
    /**
     * Protocol for listener, expects TCP or TLS
     *
     * @default - TLS if certificates are provided. TCP otherwise.
     */
    readonly protocol?: Protocol;
    /**
     * Certificate list of ACM cert ARNs
     *
     * @default - No certificates.
     */
    readonly certificates?: IListenerCertificate[];
    /**
     * SSL Policy
     *
     * @default - Current predefined security policy.
     */
    readonly sslPolicy?: SslPolicy;
}
/**
 * Properties for adding a certificate to a listener
 *
 * This interface exists for backwards compatibility.
 *
 * @deprecated Use IListenerCertificate instead
 */
export interface INetworkListenerCertificateProps extends IListenerCertificate {
}
/**
 * Properties for a Network Listener attached to a Load Balancer
 */
export interface NetworkListenerProps extends BaseNetworkListenerProps {
    /**
     * The load balancer to attach this listener to
     */
    readonly loadBalancer: INetworkLoadBalancer;
}
/**
 * Define a Network Listener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
export declare class NetworkListener extends BaseListener implements INetworkListener {
    /**
     * Import an existing listener
     */
    static fromNetworkListenerArn(scope: Construct, id: string, networkListenerArn: string): INetworkListener;
    /**
     * The load balancer this listener is attached to
     */
    private readonly loadBalancer;
    constructor(scope: Construct, id: string, props: NetworkListenerProps);
    /**
     * Load balance incoming requests to the given target groups.
     */
    addTargetGroups(_id: string, ...targetGroups: INetworkTargetGroup[]): void;
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * @returns The newly created target group
     */
    addTargets(id: string, props: AddNetworkTargetsProps): NetworkTargetGroup;
}
/**
 * Properties to reference an existing listener
 */
export interface INetworkListener extends IResource {
    /**
     * ARN of the listener
     * @attribute
     */
    readonly listenerArn: string;
}
/**
 * Properties for adding new network targets to a listener
 */
export interface AddNetworkTargetsProps {
    /**
     * The port on which the listener listens for requests.
     *
     * @default Determined from protocol if known
     */
    readonly port: number;
    /**
     * The targets to add to this target group.
     *
     * Can be `Instance`, `IPAddress`, or any self-registering load balancing
     * target. If you use either `Instance` or `IPAddress` as targets, all
     * target must be of the same type.
     */
    readonly targets?: INetworkLoadBalancerTarget[];
    /**
     * The name of the target group.
     *
     * This name must be unique per region per account, can have a maximum of
     * 32 characters, must contain only alphanumeric characters or hyphens, and
     * must not begin or end with a hyphen.
     *
     * @default Automatically generated
     */
    readonly targetGroupName?: string;
    /**
     * The amount of time for Elastic Load Balancing to wait before deregistering a target.
     *
     * The range is 0-3600 seconds.
     *
     * @default Duration.minutes(5)
     */
    readonly deregistrationDelay?: Duration;
    /**
     * Indicates whether Proxy Protocol version 2 is enabled.
     *
     * @default false
     */
    readonly proxyProtocolV2?: boolean;
    /**
     * Health check configuration
     *
     * @default No health check
     */
    readonly healthCheck?: HealthCheck;
}
