"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const network_target_group_1 = require("./network-target-group");
/**
 * Define a Network Listener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class NetworkListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        const certs = props.certificates || [];
        const proto = props.protocol || (certs.length > 0 ? enums_1.Protocol.TLS : enums_1.Protocol.TCP);
        if (NLB_PROTOCOLS.indexOf(proto) === -1) {
            throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${props.protocol}`);
        }
        if (proto === enums_1.Protocol.TLS && certs.filter(v => v != null).length === 0) {
            throw new Error('When the protocol is set to TLS, you must specify certificates');
        }
        if (proto !== enums_1.Protocol.TLS && certs.length > 0) {
            throw new Error('Protocol must be TLS when certificates have been specified');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            protocol: proto,
            port: props.port,
            sslPolicy: props.sslPolicy,
            certificates: props.certificates,
        });
        this.loadBalancer = props.loadBalancer;
        (props.defaultTargetGroups || []).forEach(this._addDefaultTargetGroup.bind(this));
    }
    /**
     * Import an existing listener
     */
    static fromNetworkListenerArn(scope, id, networkListenerArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.listenerArn = networkListenerArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Load balance incoming requests to the given target groups.
     */
    addTargetGroups(_id, ...targetGroups) {
        // New default target(s)
        for (const targetGroup of targetGroups) {
            this._addDefaultTargetGroup(targetGroup);
            targetGroup.registerListener(this);
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        if (!this.loadBalancer.vpc) {
            // tslint:disable-next-line:max-line-length
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or imported Load Balancer with specified VPC; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new network_target_group_1.NetworkTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            proxyProtocolV2: props.proxyProtocolV2,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, group);
        return group;
    }
}
exports.NetworkListener = NetworkListener;
const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
//# sourceMappingURL=data:application/json;base64,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