"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
/**
 * Define the target of a load balancer
 */
class TargetGroupBase extends cdk.Construct {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.stringValue({ produce: () => this.targetType }),
            targets: cdk.Lazy.anyValue({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.stringValue({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.anyValue({ produce: () => this.healthCheck && this.healthCheck.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.anyValue({ produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            this.node.addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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