"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// tslint:disable-next-line:max-line-length
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
exports.ifUndefined = ifUndefined;
//# sourceMappingURL=data:application/json;base64,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