"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JenkinsAction = exports.JenkinsActionType = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const action_1 = require("../action");
const jenkins_provider_1 = require("./jenkins-provider");
/**
 * The type of the Jenkins Action that determines its CodePipeline Category - Build, or Test.
 *
 * Note that a Jenkins provider, even if it has the same name,
 * must be separately registered for each type.
 *
 * @stability stable
 */
var JenkinsActionType;
(function (JenkinsActionType) {
    JenkinsActionType[JenkinsActionType["BUILD"] = 0] = "BUILD";
    JenkinsActionType[JenkinsActionType["TEST"] = 1] = "TEST";
})(JenkinsActionType = exports.JenkinsActionType || (exports.JenkinsActionType = {}));
/**
 * Jenkins build CodePipeline Action.
 *
 * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/tutorials-four-stage-pipeline.html
 * @stability stable
 */
class JenkinsAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        super({
            ...props,
            category: props.type === JenkinsActionType.BUILD
                ? codepipeline.ActionCategory.BUILD
                : codepipeline.ActionCategory.TEST,
            provider: props.jenkinsProvider.providerName,
            owner: 'Custom',
            artifactBounds: jenkins_provider_1.jenkinsArtifactsBounds,
            version: props.jenkinsProvider.version,
        });
        this.props = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(_scope, _stage, _options) {
        if (this.actionProperties.category === codepipeline.ActionCategory.BUILD) {
            this.props.jenkinsProvider._registerBuildProvider();
        }
        else {
            this.props.jenkinsProvider._registerTestProvider();
        }
        return {
            configuration: {
                ProjectName: this.props.projectName,
            },
        };
    }
}
exports.JenkinsAction = JenkinsAction;
//# sourceMappingURL=data:application/json;base64,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