# Copyright 2023 Newcastle University.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# Autogenerate examples to test the data warehouse
# Valid examples are generated by functions that begin with "valid"

from typing import Tuple, List, Optional
from random import randint, uniform, choice, random
from string import ascii_lowercase, ascii_uppercase, digits
from datetime import datetime

from data_warehouse_client.check_bounded_values import get_bounds
from data_warehouse_client.load_data import process_measurement_groups
from data_warehouse_client.multiple_mg_inserts import insert_measurement_group_instances
from data_warehouse_client.print_metadata_table import create_measurement_group_info
from data_warehouse_client.type_checks import category_ids, int_bounds, real_bounds, datetime_bounds
from data_warehouse_client.type_definitions import MeasurementType, Bounds, LoadHelperResult, Study, MeasurementGroup, \
    LoaderResult, MeasurementGroupInstance, Source, Participant, Trial


def valid_int_generator(measurement_type: MeasurementType, bounds: Bounds) -> int:
    return randint(-2147483648, +2147483647)


def valid_real_generator(measurement_type: MeasurementType, bounds: Bounds) -> float:
    return uniform(-210000.00, +210000.00)


def random_string_generator(max_length: int) -> str:
    size = randint(0, 500)
    chars = ascii_lowercase+ascii_uppercase+digits
    return ''.join(choice(chars) for _ in range(size))


def valid_text_generator(measurement_type: MeasurementType, bounds: Bounds) -> str:
    return random_string_generator(500)


def valid_datetime_generator(measurement_type: MeasurementType, bounds: Bounds) -> datetime:
    start_date = datetime(1900, 1, 1)
    end_date = datetime(2100, 1, 1)
    random_date = start_date + (end_date - start_date) * random()
    return random_date


def valid_boolean_generator(measurement_type: MeasurementType, bounds: Bounds) -> bool:
    return choice([True, False])


def valid_nominal_by_id_generator(measurement_type: MeasurementType, bounds: Bounds) -> int:
    cat_ids = category_ids(bounds)
    return choice(cat_ids[measurement_type])


def valid_ordinal_by_id_generator(measurement_type: MeasurementType, bounds: Bounds) -> int:
    cat_ids = category_ids(bounds)
    return choice(cat_ids[measurement_type])


def valid_bounded_int_generator(measurement_type: MeasurementType, bounds: Bounds) -> int:
    integer_bounds = int_bounds(bounds)
    bounds_mt = integer_bounds[measurement_type]   # get the bounds for this measurement type
    return randint(bounds_mt['minval'], bounds_mt['maxval'])


def valid_bounded_real_generator(measurement_type: MeasurementType, bounds: Bounds) -> float:
    r_bounds = real_bounds(bounds)
    bounds_mt = r_bounds[measurement_type]  # get the bounds for this measurement type
    return uniform(bounds_mt['minval'], bounds_mt['maxval'])


def valid_bounded_datetime_generator(measurement_type: MeasurementType, bounds: Bounds) -> datetime:
    dt_bounds = datetime_bounds(bounds)
    bounds_mt = dt_bounds[measurement_type]  # get the bounds for this measurement type
    start_date = bounds_mt['minval']
    end_date = bounds_mt['maxval']
    random_date = start_date + (end_date - start_date) * random()
    return random_date


def valid_external_generator(measurement_type: MeasurementType, bounds: Bounds) -> str:
    return random_string_generator(500)


def mk_random_valid_value(measurement_type, mt_info, bounds: Bounds) -> LoadHelperResult:
    # measurement_type = mt_info.key[0]
    value_type = mt_info['valtype']
    if value_type == 0:  # int
        value = valid_int_generator(measurement_type, bounds)
    elif value_type == 1:  # real
        value = valid_real_generator(measurement_type, bounds)
    elif value_type == 2:  # text
        value = valid_text_generator(measurement_type, bounds)
    elif value_type == 3:  # datetime
        value = valid_datetime_generator(measurement_type, bounds)
    elif value_type == 4:  # boolean
        value = valid_boolean_generator(measurement_type, bounds)
    elif value_type == 5:  # nominal
        value = valid_nominal_by_id_generator(measurement_type, bounds)
    elif value_type == 6:  # ordinal
        value = valid_ordinal_by_id_generator(measurement_type, bounds)
    elif value_type == 7:  # bounded int
        value = valid_bounded_int_generator(measurement_type, bounds)
    elif value_type == 8:  # bounded real
        value = valid_bounded_real_generator(measurement_type, bounds)
    elif value_type == 9:  # bounded datetime
        value = valid_bounded_datetime_generator(measurement_type, bounds)
    elif value_type == 10:  # external
        value = valid_external_generator(measurement_type, bounds)
    else:
        value = None
        print("Error in generating value")
    return True, [(measurement_type, value_type, value)], ""


def invalid_int_generator(measurement_type: MeasurementType, bounds: Bounds) -> float:
    return valid_real_generator(measurement_type, bounds)


def invalid_real_generator(measurement_type: MeasurementType, bounds: Bounds) -> str:
    return valid_text_generator(measurement_type, bounds)


def invalid_text_generator(measurement_type: MeasurementType, bounds: Bounds) -> str:
    return valid_text_generator(measurement_type, bounds)


def invalid_datetime_generator(measurement_type: MeasurementType, bounds: Bounds) -> str:
    return valid_text_generator(measurement_type, bounds)


def invalid_boolean_generator(measurement_type: MeasurementType, bounds: Bounds) -> float:
    return valid_real_generator(measurement_type, bounds)


def invalid_nominal_by_id_generator(measurement_type: MeasurementType, bounds: Bounds) -> int:
    return valid_boolean_generator(measurement_type, bounds)


def invalid_ordinal_by_id_generator(measurement_type: MeasurementType, bounds: Bounds) -> int:
    return valid_boolean_generator(measurement_type, bounds)


def invalid_bounded_int_generator(measurement_type: MeasurementType, bounds: Bounds) -> int:
    integer_bounds = int_bounds(bounds)
    bounds_mt = integer_bounds[measurement_type]   # get the bounds for this measurement type
    return randint(bounds_mt['maxval'] + 1, bounds_mt['maxval'] + 99999)


def invalid_bounded_real_generator(measurement_type: MeasurementType, bounds: Bounds) -> float:
    r_bounds = real_bounds(bounds)
    bounds_mt = r_bounds[measurement_type]  # get the bounds for this measurement type
    return uniform(bounds_mt['maxval'] + 1.0, bounds_mt['maxval'] + 999.5)


def invalid_bounded_datetime_generator(measurement_type: MeasurementType, bounds: Bounds) -> datetime:
    dt_bounds = datetime_bounds(bounds)
    bounds_mt = dt_bounds[measurement_type]  # get the bounds for this measurement type
    start_date = bounds_mt['minval']
    end_date = bounds_mt['maxval']
    random_date = end_date + start_date
    return random_date


def invalid_external_generator(measurement_type: MeasurementType, bounds: Bounds) -> int:
    return valid_int_generator(measurement_type, bounds)


def mk_random_invalid_value(measurement_type, mt_info, bounds: Bounds) -> LoadHelperResult:
    # measurement_type = mt_info.key[0]
    value_type = mt_info['valtype']
    if value_type == 0:  # int
        value = valid_int_generator(measurement_type, bounds)
    elif value_type == 1:  # real
        value = valid_real_generator(measurement_type, bounds)
    elif value_type == 2:  # text
        value = valid_text_generator(measurement_type, bounds)
    elif value_type == 3:  # datetime
        value = valid_datetime_generator(measurement_type, bounds)
    elif value_type == 4:  # boolean
        value = valid_boolean_generator(measurement_type, bounds)
    elif value_type == 5:  # nominal
        value = valid_nominal_by_id_generator(measurement_type, bounds)
    elif value_type == 6:  # ordinal
        value = valid_ordinal_by_id_generator(measurement_type, bounds)
    elif value_type == 7:  # bounded int
        value = valid_bounded_int_generator(measurement_type, bounds)
    elif value_type == 8:  # bounded real
        value = valid_bounded_real_generator(measurement_type, bounds)
    elif value_type == 9:  # bounded datetime
        value = valid_bounded_datetime_generator(measurement_type, bounds)
    elif value_type == 10:  # external
        value = valid_external_generator(measurement_type, bounds)
    else:
        value = None
        print("Error in generating value")
    return True, [(measurement_type, value_type, value)], ""  # Keep success = True and see if insert checks find prob


def valid_measurement_group_values(data_warehouse_handle,
                                   study: Study,
                                   measurement_group_id: MeasurementGroup,
                                   bounds: Bounds = None) -> LoaderResult:
    """
    Generate a complete measurement group instance with valid, random values for all the measuerements within it
    :param: data_warehouse_handle: handle of the warehouse
    :param: study: study id
    :param: measurement_group_id: measurement group identifier for the group to be generated
    :param: bounds: bounds to be respected when vlaues area randomly created
    :result: the result that a valid loader would create
    """
    if bounds is None:
        bounds: Bounds = get_bounds(data_warehouse_handle, study)  # extract bounds from warehouse
    # Retrieve information on the measurement group
    measurement_group_info = create_measurement_group_info(data_warehouse_handle, study)[measurement_group_id]
    # Retrueve information on all the measurement types in the measurement group
    measurement_type_info = measurement_group_info['message_types']
    # get the ids of all the measurement types
    mt_ids = list(map(lambda mt: mt, measurement_type_info))
    # generate random values for each measurement type in the measurement group
    load_helper_results: List[LoadHelperResult] = \
        list(map(lambda mt: mk_random_valid_value(mt, measurement_type_info[mt], bounds), mt_ids))
    return [(measurement_group_id, load_helper_results)], None, None, None, None


def load_test_data(data_warehouse_handle,
                   study: Study,
                   measurement_group_id: MeasurementGroup,
                   bounds: Bounds = None,
                   trial: Optional[Trial] = None,
                   participant: Optional[Participant] = None,
                   source: Optional[Source] = None) -> Tuple[bool, List[MeasurementGroupInstance], List[str]]:
    """
    Generate random test data for a measurement group and load a measurement group instance into the datawarehouse.
    :param data_warehouse_handle: handle to access the data warehouse
    :param study: study id in the data warehouse
    :param measurement_group_id: Measurement Group to generate test case for
    :param bounds: tuple holding bounds used to check data
    :param trial: optional trial id
    :param participant: optional participant id (must already be in the warehouse)
    :param source: optional source id
    :return: Success?, list of measurement group instance ids for the measurement groups inserted, error messages
    """
    if bounds is None:  # Generate the bounds from the data warehouse
        bounds = get_bounds(data_warehouse_handle, study)
    vals_to_load_in_msgs, time_from_data, trial_from_data, participant_from_data, source_from_data = \
        valid_measurement_group_values(data_warehouse_handle, study, measurement_group_id, bounds=bounds)
    successful, all_mgs_and_triples, combined_error_messages = process_measurement_groups(vals_to_load_in_msgs)
    if successful:  # all values are ready to be inserted into the data warehouse
        if trial_from_data is None:  # if the loader function has returned a trial id then use it
            trial_to_insert = trial
        else:
            trial_to_insert = trial_from_data
        if participant_from_data is None:  # if the loader function has returned a participant id then use it
            participant_to_insert = participant
        else:
            participant_to_insert = participant_from_data
        if source_from_data is None:  # if the loader function has returned a source id then use it
            source_to_insert = source
        else:
            source_to_insert = source_from_data
        # try to insert the measurement group instances
        successful_inserts, message_group_instance_ids, error_message = \
            insert_measurement_group_instances(
                data_warehouse_handle, study, all_mgs_and_triples, bounds=bounds,
                time=time_from_data, trial=trial_to_insert, participant=participant_to_insert, source=source_to_insert)
        return successful_inserts, message_group_instance_ids, error_message
    else:
        return False, [], combined_error_messages   # error(s) detected in the data to be loaded


def load_test_data_for_measurement_groups(
        data_warehouse_handle,
        study: Study,
        measurement_groups: List[MeasurementGroup],
        bounds: Bounds = None,
        trial: Optional[Trial] = None,
        participant: Optional[Participant] = None,
        source: Optional[Source] = None) -> \
            List[Tuple[MeasurementGroup, Tuple[bool, List[MeasurementGroupInstance], List[str]]]]:
    if bounds is None:  # Generate the bounds from the data warehouse
        bounds = get_bounds(data_warehouse_handle, study)
    # insert a random, valid measurement group instance in the data warehouse for ecah measurement group
    result = list(map(lambda mg: (mg, load_test_data(data_warehouse_handle, study, mg, bounds, trial, participant,
                                                     source)),
                      measurement_groups))
    return result


def load_test_data_for_all_measurement_groups(
        data_warehouse_handle,
        study: Study,
        bounds: Bounds = None,
        trial: Optional[Trial] = None,
        participant: Optional[Participant] = None,
        source: Optional[Source] = None) -> \
            List[Tuple[MeasurementGroup, Tuple[bool, List[MeasurementGroupInstance], List[str]]]]:
    # get info on all measurement groups
    measurement_group_info = create_measurement_group_info(data_warehouse_handle, study)
    # get a list of all measurement groups
    all_measurement_groups = measurement_group_info.keys()
    return load_test_data_for_measurement_groups(data_warehouse_handle, study, all_measurement_groups, bounds, trial,
                                                 participant, source)
