# Copyright 2023 Newcastle University.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
from datetime import datetime

# Autogenerate examples to test the data warehouse
# Valid examples are generated by functions that begin with "valid"

import pytest

from data_warehouse_client import auto_generate_test_cases
from data_warehouse_client.data_warehouse import DataWarehouse
from data_warehouse_client.print_metadata_table import create_measurement_group_info
from data_warehouse_client.type_checks import check_int, check_real, check_string, check_datetime, check_boolean, \
    check_nominal, check_ordinal, check_bounded_int, check_bounded_real, check_bounded_datetime, check_external
from data_warehouse_client.type_definitions import Bounds


@pytest.fixture()
def bounds_ex1() -> Bounds:
    int_bounds = {1: {'minval': 0, 'maxval': 100}}
    real_bounds = {2: {'minval': 0.0, 'maxval': 100.0}}
    start_date = datetime(1900, 1, 1)
    end_date = datetime(2100, 1, 1)
    datetime_bounds = {3: {'minval': start_date, 'maxval': end_date}}
    category_ids = {4: [1, 2, 3, 4, 5]}
    inverse_categy_ids = {5: {'first': 1, 'second': 2, 'third': 3}}
    return int_bounds, real_bounds, datetime_bounds, category_ids, inverse_categy_ids


def test_valid_int_generator(bounds_ex1: Bounds):
    i = auto_generate_test_cases.valid_int_generator(10, bounds_ex1)
    assert check_int(i)


def test_valid_real_generator(bounds_ex1: Bounds):
    r = auto_generate_test_cases.valid_real_generator(11, bounds_ex1)
    assert check_real(r)


def test_valid_text_generator(bounds_ex1: Bounds):
    t = auto_generate_test_cases.valid_text_generator(12, bounds_ex1)
    assert check_string(t)


def test_valid_datetime_generator(bounds_ex1: Bounds):
    d = auto_generate_test_cases.valid_datetime_generator(13, bounds_ex1)
    assert check_datetime(d)


def test_valid_boolean_generator(bounds_ex1: Bounds):
    d = auto_generate_test_cases.valid_boolean_generator(14, bounds_ex1)
    assert check_boolean(d)


def test_valid_nominal_by_id(bounds_ex1: Bounds):
    ident = auto_generate_test_cases.valid_nominal_by_id_generator(4, bounds_ex1)
    assert check_nominal(ident, 4, bounds_ex1)


def test_valid_ordinal_by_id(bounds_ex1: Bounds):
    ident = auto_generate_test_cases.valid_ordinal_by_id_generator(4, bounds_ex1)
    assert check_ordinal(ident, 4, bounds_ex1)


def test_valid_bounded_int(bounds_ex1: Bounds):
    i = auto_generate_test_cases.valid_bounded_int_generator(1, bounds_ex1)
    assert check_bounded_int(i, 1, bounds_ex1)


def test_valid_bounded_real(bounds_ex1: Bounds):
    r = auto_generate_test_cases.valid_bounded_real_generator(2, bounds_ex1)
    assert check_bounded_real(r, 2, bounds_ex1)


def test_valid_bounded_datetime(bounds_ex1: Bounds):
    d = auto_generate_test_cases.valid_datetime_generator(3, bounds_ex1)
    assert check_bounded_datetime(d, 3, bounds_ex1)


def test_valid_bounded_external(bounds_ex1: Bounds):
    e = auto_generate_test_cases.valid_external_generator(15, bounds_ex1)
    assert check_external(e, 15, bounds_ex1)


@pytest.fixture()
def test_study():
    return 1


@pytest.fixture()
def database_name():
    return "osm_dw"


@pytest.fixture()
def credentials_full_access_file_name():
    return "dw-credentials-full-access.json"


@pytest.fixture()
def mk_dw_handle_full_access(credentials_full_access_file_name, database_name):
    data_warehouse_handle = DataWarehouse(credentials_full_access_file_name, database_name)
    yield data_warehouse_handle


def test_load_test_data_for_all_measurement_groups(mk_dw_handle_full_access, test_study):
    data_warehouse_handle = mk_dw_handle_full_access
    mg_info = create_measurement_group_info(data_warehouse_handle, test_study)
    number_of_measurement_groups = len(mg_info)
    res = auto_generate_test_cases.load_test_data_for_all_measurement_groups(data_warehouse_handle, test_study)
    assert len(res) == number_of_measurement_groups
