"""Markdown output formatter for ontology statistics."""

from typing import Optional

from rdflib import Graph

from rdf_construct.stats.collector import OntologyStats
from rdf_construct.stats.comparator import ComparisonResult


def _format_pct(value: float) -> str:
    """Format a percentage value for display."""
    return f"{value * 100:.1f}%"


def format_markdown_stats(stats: OntologyStats, graph: Optional[Graph] = None) -> str:
    """Format ontology statistics as Markdown table.

    Designed for embedding in README files or documentation.

    Args:
        stats: The statistics to format.
        graph: Optional graph for CURIE formatting.

    Returns:
        Markdown string representation.
    """
    lines = []

    lines.append(f"## Ontology Statistics")
    lines.append("")
    lines.append(f"**Source:** `{stats.source}`")
    lines.append("")

    # Summary table
    lines.append("| Metric | Value |")
    lines.append("|--------|-------|")
    lines.append(f"| Classes | {stats.basic.classes} |")
    lines.append(f"| Object Properties | {stats.basic.object_properties} |")
    lines.append(f"| Datatype Properties | {stats.basic.datatype_properties} |")
    lines.append(f"| Max Hierarchy Depth | {stats.hierarchy.max_depth} |")
    lines.append(f"| Documentation Coverage | {_format_pct(stats.documentation.classes_documented_pct)} |")
    lines.append("")

    # Detailed sections
    lines.append("### Structure")
    lines.append("")
    lines.append("| Metric | Value |")
    lines.append("|--------|-------|")
    lines.append(f"| Total Triples | {stats.basic.triples:,} |")
    lines.append(f"| Root Classes | {stats.hierarchy.root_classes} |")
    lines.append(f"| Leaf Classes | {stats.hierarchy.leaf_classes} |")
    lines.append(f"| Avg Branching Factor | {stats.hierarchy.avg_branching:.1f} |")
    lines.append(f"| Orphan Classes | {stats.hierarchy.orphan_classes} ({_format_pct(stats.hierarchy.orphan_rate)}) |")
    lines.append("")

    lines.append("### Properties")
    lines.append("")
    lines.append("| Metric | Value |")
    lines.append("|--------|-------|")
    lines.append(f"| With Domain | {stats.properties.with_domain} ({_format_pct(stats.properties.domain_coverage)}) |")
    lines.append(f"| With Range | {stats.properties.with_range} ({_format_pct(stats.properties.range_coverage)}) |")
    lines.append(f"| Functional | {stats.properties.functional} |")
    lines.append(f"| Symmetric | {stats.properties.symmetric} |")
    lines.append("")

    lines.append("### Documentation")
    lines.append("")
    lines.append("| Metric | Value |")
    lines.append("|--------|-------|")
    lines.append(f"| Classes Labelled | {stats.documentation.classes_labelled} ({_format_pct(stats.documentation.classes_labelled_pct)}) |")
    lines.append(f"| Classes Documented | {stats.documentation.classes_documented} ({_format_pct(stats.documentation.classes_documented_pct)}) |")
    lines.append(f"| Properties Labelled | {stats.documentation.properties_labelled} ({_format_pct(stats.documentation.properties_labelled_pct)}) |")
    lines.append("")

    lines.append("*Generated by rdf-construct stats*")

    return "\n".join(lines)


def format_markdown_comparison(
    comparison: ComparisonResult,
    graph: Optional[Graph] = None,
) -> str:
    """Format comparison results as Markdown.

    Args:
        comparison: The comparison result to format.
        graph: Optional graph for CURIE formatting.

    Returns:
        Markdown string representation.
    """
    lines = []

    lines.append(f"## Ontology Comparison")
    lines.append("")
    lines.append(f"**Old:** `{comparison.old_source}`")
    lines.append(f"**New:** `{comparison.new_source}`")
    lines.append("")

    if not comparison.changes:
        lines.append("No changes detected.")
        return "\n".join(lines)

    # Summary
    lines.append(f"> {comparison.summary}")
    lines.append("")

    # Changes table
    lines.append("| Category | Metric | Old | New | Change |")
    lines.append("|----------|--------|-----|-----|--------|")

    for change in comparison.changes:
        category = change.category.title()
        metric = change.metric.replace("_", " ").title()
        old_val = _format_value(change.old_value)
        new_val = _format_value(change.new_value)

        if change.delta is not None:
            if change.pct_change is not None:
                delta_str = f"{change.delta:+g} ({change.pct_change:+.1f}%)"
            else:
                delta_str = f"{change.delta:+g}"
        else:
            delta_str = "-"

        # Add indicator
        if change.improved is True:
            delta_str += " ✓"
        elif change.improved is False:
            delta_str += " ⚠"

        lines.append(f"| {category} | {metric} | {old_val} | {new_val} | {delta_str} |")

    lines.append("")
    lines.append("*Generated by rdf-construct stats --compare*")

    return "\n".join(lines)


def _format_value(value: float | int | str | None) -> str:
    """Format a metric value for display."""
    if value is None:
        return "-"
    if isinstance(value, float):
        if value < 1:
            # Probably a percentage/rate
            return f"{value * 100:.1f}%"
        return f"{value:.2f}"
    if isinstance(value, int):
        return f"{value:,}"
    return str(value)
