#!/usr/bin/env python
# coding: utf-8
"""script that makes source data .csv files for figures
that plot learning curves with error as a function of training set size,
with and without clean-up transformations applied to predictions

Note this script expects directories with the following structure:
results/Bengalese_Finches  # <-- you want to specify this path as `results-root` argument when calling this script
└── learncurve  # <-- you want to specify these sub-directories as 'experiment-dirs'
    ├── bird1  # <-- animal IDs
    │   └──results_210409_092101  # <-- results directories generated by runs of `vak learncurve`
    ├── bird2
    │   └──results_210409_011292
    ├ ...

"""
from argparse import ArgumentParser
import configparser  # used to load 'min_segment_dur.ini'

from pathlib import Path

import pandas as pd
import pyprojroot

import article


def main(results_root,
         experiment_dirs,
         segment_params_ini,
         csv_filename,
         to_annot):
    # import crowsetta here to avoid circular import
    import crowsetta

    results_root = Path(results_root)
    if not results_root.exists():
        raise NotADirectoryError(
            f'results-root not found: {results_root}'
        )

    experiment_dir_paths = []
    for experiment_dir in experiment_dirs:
        experiment_dir_path = results_root / experiment_dir
        if not experiment_dir_path.exists():
            raise NotADirectoryError(
                f'experiment directory not found: {experiment_dir_path}'
            )
        experiment_dir_paths.append(experiment_dir_path)

    for experiment_dir_path in experiment_dir_paths:
        print(
            f're-running learning curve for experiment directory: {experiment_dir_path}'
        )
        indiv_roots = sorted(
            [subdir
             for subdir in experiment_dir_path.iterdir()
             if subdir.is_dir() and not subdir.name.startswith('.')]  # 'and not' to avoid e.g. .ipynb_checkpoints
        )

        # get minimum segment durations to use for clean up. Fail early if they're not there.
        config = configparser.ConfigParser()
        config.optionxform = lambda option: option  # monkeypatch optionxform so it doesn't lowercase animal IDs
        config.read(Path(segment_params_ini).expanduser().resolve())
        min_segment_durs = {k: float(v) for k, v in config['min_segment_dur'].items()}
        for indiv_root in indiv_roots:
            if indiv_root.name not in min_segment_durs:
                raise ValueError(
                    f"could not find a minimum segment duration for animal id: {indiv_root.name}. "
                    f"Found the following animal IDs in the min_segment_dur.ini file: {min_segment_durs.keys()}"
                )

        all_eval_dfs = []
        if to_annot:
            # make dir we will save annotations in, we actually save the files below
            annot_root = results_root / 'annotations'
            annot_root.mkdir(exist_ok=True)

        for indiv_root in indiv_roots:
            results_dirs = sorted(indiv_root.glob('results_*'))
            most_recent_results = results_dirs[-1]
            min_segment_dur = min_segment_durs[indiv_root.name]
            (eval_dfs,
             learncurve_annots) = article.eval.learncurve_with_transforms(previous_run_path=most_recent_results,
                                                                          min_segment_dur=min_segment_dur,
                                                                          to_annot=to_annot)

            if to_annot:
                csv_prefix = f'{indiv_root.name}-{most_recent_results.name}-'
                for train_dur, replicate_num, annots_by_cleanup in learncurve_annots:  # unpack named tuple
                    for cleanup_type, annots_list in annots_by_cleanup.items():
                        csv_fname = csv_prefix + f'{train_dur}-{replicate_num}-cleanup-{cleanup_type}.csv'
                        csv_path = annot_root / csv_fname
                        crowsetta.csv.annot2csv(annots_list, str(csv_path))

            eval_dfs['avg_error'] = 1 - eval_dfs['avg_acc']
            eval_dfs['animal_id'] = indiv_root.name
            all_eval_dfs.append(eval_dfs)

            # save the whole thing after every animal_id,
            # so we get some result even if there's a crash
            curve_df = pd.concat(all_eval_dfs)
            curve_df['animal_id'] = curve_df['animal_id'].astype('category')
            curve_df['avg_error'] = curve_df['avg_error'] * 100

            # add 'train_set_dur_ind' column that maps train set durations to consecutive integers
            # so we can plot with those integers as the xticks, but then change the xticklabels to the actual values
            # -- this lets us avoid having the labels overlap when the training set durations are close to each other
            # e.g., 30 and 45
            train_set_durs = sorted(curve_df['train_set_dur'].unique())
            dur_int_map = dict(zip(train_set_durs, range(len(train_set_durs))))
            curve_df['train_set_dur_ind'] = curve_df['train_set_dur'].map(dur_int_map)

            # save the whole thing after every animal_id !!! (like I just said above)
            csv_path = experiment_dir_path.joinpath(csv_filename)
            curve_df.to_csv(csv_path, index=False)


PROJ_ROOT = pyprojroot.here()
BF_RESULTS_ROOT = PROJ_ROOT / 'results' / 'Bengalese_Finches'
EXPT_DIRS = (
    'learncurve',
    'hidden_size/hidden_size_16',
    'hidden_size/hidden_size_64',
    'window_size/window_size_88',
    'window_size/window_size_352',
)
SEGMENT_PARAMS_INI = PROJ_ROOT / 'data' / 'configs' / 'segment_params.ini'


def get_parser():
    parser = ArgumentParser()
    parser.add_argument('--results-root',
                        help=("root directory for all experiments from a species, "
                              "e.g. './results/Bengalese_Finches'"),
                        default=BF_RESULTS_ROOT)
    parser.add_argument('--experiment-dirs',
                        nargs='+',
                        default=EXPT_DIRS,
                        help=("space-separated list of one or more director names "
                              "within 'results-root', e.g. 'learncurve window_size hidden_size'. "
                              "Each directory within an 'experiment-dir' "
                              "will have subdirectories that are individuals from dataset, "
                              "and each subdirectory contains results folders from a run "
                              "of `vak learncurve`"))
    parser.add_argument('--segment_params_ini',
                        help=("path to .ini file with segmenting parameters "
                              "for audio files from each animal"),
                        default=SEGMENT_PARAMS_INI)
    parser.add_argument('--csv-filename',
                        help='filename of .csv that will be saved by this script in results_root',
                        default='error_across_birds_with_cleanup.csv')
    parser.add_argument('--to-annot',
                        help=("if this option is added, predictions will be converted to annotations "
                              "and then saved in an 'annotations' directory in 'results_root'. "
                              "(If the option is not added, then 'to_annot' defaults to False.)"),
                        action='store_true',
                        )
    return parser


if __name__ == '__main__':
    parser = get_parser()
    args = parser.parse_args()
    main(results_root=args.results_root,
         experiment_dirs=args.experiment_dirs,
         segment_params_ini=args.segment_params_ini,
         csv_filename=args.csv_filename,
         to_annot=args.to_annot)
