"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScalableTableAttribute = void 0;
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
/**
 * A scalable table attribute
 */
class ScalableTableAttribute extends appscaling.BaseScalableAttribute {
    constructor() {
        super(...arguments);
        this.scalingPolicyCreated = false;
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        this.scalingPolicyCreated = true;
        super.doScaleOnSchedule(id, action);
    }
    /**
     * Scale out or in to keep utilization at a given level
     */
    scaleOnUtilization(props) {
        if (props.targetUtilizationPercent < 10 || props.targetUtilizationPercent > 90) {
            // eslint-disable-next-line max-len
            throw new RangeError(`targetUtilizationPercent for DynamoDB scaling must be between 10 and 90 percent, got: ${props.targetUtilizationPercent}`);
        }
        this.scalingPolicyCreated = true;
        const predefinedMetric = this.props.dimension.indexOf('ReadCapacity') === -1
            ? appscaling.PredefinedMetric.DYANMODB_WRITE_CAPACITY_UTILIZATION
            : appscaling.PredefinedMetric.DYNAMODB_READ_CAPACITY_UTILIZATION;
        super.doScaleToTrackMetric('Tracking', {
            policyName: props.policyName,
            disableScaleIn: props.disableScaleIn,
            scaleInCooldown: props.scaleInCooldown,
            scaleOutCooldown: props.scaleOutCooldown,
            targetValue: props.targetUtilizationPercent,
            predefinedMetric,
        });
    }
    /** @internal */
    get _scalingPolicyCreated() {
        return this.scalingPolicyCreated;
    }
}
exports.ScalableTableAttribute = ScalableTableAttribute;
//# sourceMappingURL=data:application/json;base64,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