"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *
 */
class Source {
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: () => ({ bucket, zipObjectKey }),
        };
    }
    /**
     * Uses a local asset as the deployment source.
     * @param path The path to a local .zip file or a directory
     */
    static asset(path, options) {
        return {
            bind(context) {
                let id = 1;
                while (context.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(context, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey,
                };
            },
        };
    }
    constructor() { }
}
exports.Source = Source;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic291cmNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic291cmNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQ0Esb0RBQW9EO0FBMEJwRDs7Ozs7Ozs7O0dBU0c7QUFDSCxNQUFhLE1BQU07SUFDakI7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBa0IsRUFBRSxZQUFvQjtRQUMzRCxPQUFPO1lBQ0wsSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsRUFBRSxNQUFNLEVBQUUsWUFBWSxFQUFFLENBQUM7U0FDdkMsQ0FBQztJQUNKLENBQUM7SUFFRDs7O09BR0c7SUFDSSxNQUFNLENBQUMsS0FBSyxDQUFDLElBQVksRUFBRSxPQUFnQztRQUNoRSxPQUFPO1lBQ0wsSUFBSSxDQUFDLE9BQXNCO2dCQUN6QixJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7Z0JBQ1gsT0FBTyxPQUFPLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsRUFBRSxDQUFDLEVBQUU7b0JBQzlDLEVBQUUsRUFBRSxDQUFDO2lCQUNOO2dCQUNELE1BQU0sS0FBSyxHQUFHLElBQUksU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsUUFBUSxFQUFFLEVBQUUsRUFBRTtvQkFDdkQsSUFBSTtvQkFDSixHQUFHLE9BQU87aUJBQ1gsQ0FBQyxDQUFDO2dCQUNILElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFO29CQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLHNEQUFzRCxDQUFDLENBQUM7aUJBQ3pFO2dCQUNELE9BQU87b0JBQ0wsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO29CQUNwQixZQUFZLEVBQUUsS0FBSyxDQUFDLFdBQVc7aUJBQ2hDLENBQUM7WUFDSixDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRCxnQkFBd0IsQ0FBQztDQUMxQjtBQXZDRCx3QkF1Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0ICogYXMgczNfYXNzZXRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMy1hc3NldHMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5leHBvcnQgaW50ZXJmYWNlIFNvdXJjZUNvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgc291cmNlIGJ1Y2tldCB0byBkZXBsb3kgZnJvbS5cbiAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldDogczMuSUJ1Y2tldDtcblxuICAvKipcbiAgICogQW4gUzMgb2JqZWN0IGtleSBpbiB0aGUgc291cmNlIGJ1Y2tldCB0aGF0IHBvaW50cyB0byBhIHppcCBmaWxlLlxuICAgKi9cbiAgcmVhZG9ubHkgemlwT2JqZWN0S2V5OiBzdHJpbmc7XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIHNvdXJjZSBmb3IgYnVja2V0IGRlcGxveW1lbnRzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElTb3VyY2Uge1xuICAvKipcbiAgICogQmluZHMgdGhlIHNvdXJjZSB0byBhIGJ1Y2tldCBkZXBsb3ltZW50LlxuICAgKiBAcGFyYW0gY29udGV4dCBUaGUgY29uc3RydWN0IHRyZWUgY29udGV4dC5cbiAgICovXG4gIGJpbmQoY29udGV4dDogY2RrLkNvbnN0cnVjdCk6IFNvdXJjZUNvbmZpZztcbn1cblxuLyoqXG4gKiBTcGVjaWZpZXMgYnVja2V0IGRlcGxveW1lbnQgc291cmNlLlxuICpcbiAqIFVzYWdlOlxuICpcbiAqICAgICBTb3VyY2UuYnVja2V0KGJ1Y2tldCwga2V5KVxuICogICAgIFNvdXJjZS5hc3NldCgnL2xvY2FsL3BhdGgvdG8vZGlyZWN0b3J5JylcbiAqICAgICBTb3VyY2UuYXNzZXQoJy9sb2NhbC9wYXRoL3RvL2EvZmlsZS56aXAnKVxuICpcbiAqL1xuZXhwb3J0IGNsYXNzIFNvdXJjZSB7XG4gIC8qKlxuICAgKiBVc2VzIGEgLnppcCBmaWxlIHN0b3JlZCBpbiBhbiBTMyBidWNrZXQgYXMgdGhlIHNvdXJjZSBmb3IgdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldCBjb250ZW50cy5cbiAgICogQHBhcmFtIGJ1Y2tldCBUaGUgUzMgQnVja2V0XG4gICAqIEBwYXJhbSB6aXBPYmplY3RLZXkgVGhlIFMzIG9iamVjdCBrZXkgb2YgdGhlIHppcCBmaWxlIHdpdGggY29udGVudHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYnVja2V0KGJ1Y2tldDogczMuSUJ1Y2tldCwgemlwT2JqZWN0S2V5OiBzdHJpbmcpOiBJU291cmNlIHtcbiAgICByZXR1cm4ge1xuICAgICAgYmluZDogKCkgPT4gKHsgYnVja2V0LCB6aXBPYmplY3RLZXkgfSksXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVc2VzIGEgbG9jYWwgYXNzZXQgYXMgdGhlIGRlcGxveW1lbnQgc291cmNlLlxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcGF0aCB0byBhIGxvY2FsIC56aXAgZmlsZSBvciBhIGRpcmVjdG9yeVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhc3NldChwYXRoOiBzdHJpbmcsIG9wdGlvbnM/OiBzM19hc3NldHMuQXNzZXRPcHRpb25zKTogSVNvdXJjZSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGJpbmQoY29udGV4dDogY2RrLkNvbnN0cnVjdCk6IFNvdXJjZUNvbmZpZyB7XG4gICAgICAgIGxldCBpZCA9IDE7XG4gICAgICAgIHdoaWxlIChjb250ZXh0Lm5vZGUudHJ5RmluZENoaWxkKGBBc3NldCR7aWR9YCkpIHtcbiAgICAgICAgICBpZCsrO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGFzc2V0ID0gbmV3IHMzX2Fzc2V0cy5Bc3NldChjb250ZXh0LCBgQXNzZXQke2lkfWAsIHtcbiAgICAgICAgICBwYXRoLFxuICAgICAgICAgIC4uLm9wdGlvbnMsXG4gICAgICAgIH0pO1xuICAgICAgICBpZiAoIWFzc2V0LmlzWmlwQXJjaGl2ZSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQXNzZXQgcGF0aCBtdXN0IGJlIGVpdGhlciBhIC56aXAgZmlsZSBvciBhIGRpcmVjdG9yeScpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgYnVja2V0OiBhc3NldC5idWNrZXQsXG4gICAgICAgICAgemlwT2JqZWN0S2V5OiBhc3NldC5zM09iamVjdEtleSxcbiAgICAgICAgfTtcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoKSB7IH1cbn1cbiJdfQ==