"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const type_1 = require("./private/type");
const assert_1 = require("./vendored/assert");
/**
 * (experimental) Partial and special matching during template assertions.
 *
 * @experimental
 */
class Match {
    /**
     * (experimental) Use this matcher in the place of a field's value, if the field must not be present.
     *
     * @experimental
     */
    static absentProperty() {
        return assert_1.ABSENT;
    }
    /**
     * (experimental) Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must be in the same order as would be found.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * (experimental) Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must match exactly and in order.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * (experimental) Deep exact matching of the specified pattern to the target.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * (experimental) Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * (experimental) Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must match exactly with the target.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
    /**
     * (experimental) Matches any target which does NOT follow the specified pattern.
     *
     * @param pattern the pattern to NOT match.
     * @experimental
     */
    static not(pattern) {
        return new NotMatch('not', pattern);
    }
    /**
     * (experimental) Matches any string-encoded JSON and applies the specified pattern after parsing it.
     *
     * @param pattern the pattern to match after parsing the encoded JSON.
     * @experimental
     */
    static serializedJson(pattern) {
        return new SerializedJson('serializedJson', pattern);
    }
    /**
     * (experimental) Matches any non-null value at the target.
     *
     * @experimental
     */
    static anyValue() {
        return new AnyMatch('anyValue');
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "@aws-cdk/assertions.Match", version: "1.125.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = (_b = options.partialObjects) !== null && _b !== void 0 ? _b : false;
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult(actual);
        if (typeof this.pattern !== typeof actual) {
            result.push(this, [], `Expected type ${typeof this.pattern} but received ${type_1.getType(actual)}`);
            return result;
        }
        if (this.pattern === assert_1.ABSENT) {
            throw new Error('absentProperty() can only be used in an object matcher');
        }
        if (actual !== this.pattern) {
            result.push(this, [], `Expected ${this.pattern} but received ${actual}`);
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.subsequence = (_b = options.subsequence) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected type array but received ${type_1.getType(actual)}`);
        }
        if (!this.subsequence && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected array of length ${this.pattern.length} but received ${actual.length}`);
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult(actual);
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement) ? patternElement : new LiteralMatch(this.name, patternElement);
            if (this.subsequence && matcher instanceof AnyMatch) {
                // array subsequence matcher is not compatible with anyValue() matcher. They don't make sense to be used together.
                throw new Error('The Matcher anyValue() cannot be nested within arrayWith()');
            }
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.subsequence || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.push(this, [], `Missing element${element}at pattern index ${patternIdx}`);
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.partial) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected type object but received ${type_1.getType(actual)}`);
        }
        const result = new matcher_1.MatchResult(actual);
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.push(this, [`/${a}`], 'Unexpected key');
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (patternVal === assert_1.ABSENT) {
                if (patternKey in actual) {
                    result.push(this, [`/${patternKey}`], 'Key should be absent');
                }
                continue;
            }
            if (!(patternKey in actual)) {
                result.push(this, [`/${patternKey}`], 'Missing key');
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
class SerializedJson extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    ;
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (type_1.getType(actual) !== 'string') {
            result.push(this, [], `Expected JSON as a string but found ${type_1.getType(actual)}`);
            return result;
        }
        let parsed;
        try {
            parsed = JSON.parse(actual);
        }
        catch (err) {
            if (err instanceof SyntaxError) {
                result.push(this, [], `Invalid JSON string: ${actual}`);
                return result;
            }
            else {
                throw err;
            }
        }
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(parsed);
        result.compose(`(${this.name})`, innerResult);
        return result;
    }
}
class NotMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(actual);
        const result = new matcher_1.MatchResult(actual);
        if (innerResult.failCount === 0) {
            result.push(this, [], `Found unexpected match: ${JSON.stringify(actual, undefined, 2)}`);
        }
        return result;
    }
}
class AnyMatch extends matcher_1.Matcher {
    constructor(name) {
        super();
        this.name = name;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (actual == null) {
            result.push(this, [], 'Expected a value but found none');
        }
        return result;
    }
}
//# sourceMappingURL=data:application/json;base64,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