"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cannedMetricsForService = void 0;
const canned_metrics_schema_1 = require("./canned-metrics/canned-metrics-schema");
/**
 * Return the list of canned metrics for the given service
 */
function cannedMetricsForService(cloudFormationNamespace) {
    var _a;
    // One metricTemplate has a single set of dimensions, but the same metric NAME
    // may occur in multiple metricTemplates (if it has multiple sets of dimensions)
    const metricTemplates = (_a = cannedMetricsIndex()[cloudFormationNamespace]) !== null && _a !== void 0 ? _a : [];
    // First construct almost what we need, but with a single dimension per metric
    const metricsWithDuplicates = flatMap(metricTemplates, metricSet => {
        const dimensions = metricSet.dimensions.map(d => d.dimensionName);
        return metricSet.metrics.map(metric => ({
            namespace: metricSet.namespace,
            dimensions,
            metricName: metric.name,
            defaultStat: metric.defaultStat,
        }));
    });
    // Then combine the dimensions for the same metrics into a single list
    return groupBy(metricsWithDuplicates, m => `${m.namespace}/${m.metricName}`).map(metrics => ({
        namespace: metrics[0].namespace,
        metricName: metrics[0].metricName,
        defaultStat: metrics[0].defaultStat,
        dimensions: Array.from(dedupeStringLists(metrics.map(m => m.dimensions))),
    }));
}
exports.cannedMetricsForService = cannedMetricsForService;
let cannedMetricsCache;
/**
 * Load the canned metrics file and process it into an index, grouped by service namespace
 */
function cannedMetricsIndex() {
    var _a;
    if (cannedMetricsCache === undefined) {
        cannedMetricsCache = {};
        for (const group of canned_metrics_schema_1.loadCannedMetricsFile()) {
            for (const metricTemplate of group.metricTemplates) {
                const [aws, service] = metricTemplate.resourceType.split('::');
                const serviceKey = [aws, service].join('::');
                ((_a = cannedMetricsCache[serviceKey]) !== null && _a !== void 0 ? _a : (cannedMetricsCache[serviceKey] = [])).push(metricTemplate);
            }
        }
    }
    return cannedMetricsCache;
}
function flatMap(xs, fn) {
    return Array.prototype.concat.apply([], xs.map(fn));
}
function groupBy(xs, keyFn) {
    const obj = {};
    for (const x of xs) {
        const key = keyFn(x);
        if (key in obj) {
            obj[key].push(x);
        }
        else {
            obj[key] = [x];
        }
    }
    return Object.values(obj);
}
function* dedupeStringLists(xs) {
    const seen = new Set();
    for (const x of xs) {
        x.sort();
        const key = `${x.join(',')}`;
        if (!seen.has(key)) {
            yield x;
        }
        seen.add(key);
    }
}
//# sourceMappingURL=data:application/json;base64,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