"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isPropertyDifference = exports.ResourceDifference = exports.ResourceImpact = exports.ParameterDifference = exports.OutputDifference = exports.MetadataDifference = exports.MappingDifference = exports.ConditionDifference = exports.DifferenceCollection = exports.PropertyDifference = exports.Difference = exports.TemplateDiff = void 0;
const assert_1 = require("assert");
const cfnspec = require("../../cfnspec/lib");
const iam_changes_1 = require("../iam/iam-changes");
const security_group_changes_1 = require("../network/security-group-changes");
const util_1 = require("./util");
/** Semantic differences between two CloudFormation templates. */
class TemplateDiff {
    constructor(args) {
        if (args.awsTemplateFormatVersion !== undefined) {
            this.awsTemplateFormatVersion = args.awsTemplateFormatVersion;
        }
        if (args.description !== undefined) {
            this.description = args.description;
        }
        if (args.transform !== undefined) {
            this.transform = args.transform;
        }
        this.conditions = args.conditions || new DifferenceCollection({});
        this.mappings = args.mappings || new DifferenceCollection({});
        this.metadata = args.metadata || new DifferenceCollection({});
        this.outputs = args.outputs || new DifferenceCollection({});
        this.parameters = args.parameters || new DifferenceCollection({});
        this.resources = args.resources || new DifferenceCollection({});
        this.unknown = args.unknown || new DifferenceCollection({});
        this.iamChanges = new iam_changes_1.IamChanges({
            propertyChanges: this.scrutinizablePropertyChanges(iam_changes_1.IamChanges.IamPropertyScrutinies),
            resourceChanges: this.scrutinizableResourceChanges(iam_changes_1.IamChanges.IamResourceScrutinies),
        });
        this.securityGroupChanges = new security_group_changes_1.SecurityGroupChanges({
            egressRulePropertyChanges: this.scrutinizablePropertyChanges([cfnspec.schema.PropertyScrutinyType.EgressRules]),
            ingressRulePropertyChanges: this.scrutinizablePropertyChanges([cfnspec.schema.PropertyScrutinyType.IngressRules]),
            egressRuleResourceChanges: this.scrutinizableResourceChanges([cfnspec.schema.ResourceScrutinyType.EgressRuleResource]),
            ingressRuleResourceChanges: this.scrutinizableResourceChanges([cfnspec.schema.ResourceScrutinyType.IngressRuleResource]),
        });
    }
    get differenceCount() {
        let count = 0;
        if (this.awsTemplateFormatVersion !== undefined) {
            count += 1;
        }
        if (this.description !== undefined) {
            count += 1;
        }
        if (this.transform !== undefined) {
            count += 1;
        }
        count += this.conditions.differenceCount;
        count += this.mappings.differenceCount;
        count += this.metadata.differenceCount;
        count += this.outputs.differenceCount;
        count += this.parameters.differenceCount;
        count += this.resources.differenceCount;
        count += this.unknown.differenceCount;
        return count;
    }
    get isEmpty() {
        return this.differenceCount === 0;
    }
    /**
     * Return true if any of the permissions objects involve a broadening of permissions
     */
    get permissionsBroadened() {
        return this.iamChanges.permissionsBroadened || this.securityGroupChanges.rulesAdded;
    }
    /**
     * Return true if any of the permissions objects have changed
     */
    get permissionsAnyChanges() {
        return this.iamChanges.hasChanges || this.securityGroupChanges.hasChanges;
    }
    /**
     * Return all property changes of a given scrutiny type
     *
     * We don't just look at property updates; we also look at resource additions and deletions (in which
     * case there is no further detail on property values), and resource type changes.
     */
    scrutinizablePropertyChanges(scrutinyTypes) {
        const ret = new Array();
        for (const [resourceLogicalId, resourceChange] of Object.entries(this.resources.changes)) {
            if (!resourceChange) {
                continue;
            }
            const props = cfnspec.scrutinizablePropertyNames(resourceChange.newResourceType, scrutinyTypes);
            for (const propertyName of props) {
                ret.push({
                    resourceLogicalId,
                    propertyName,
                    resourceType: resourceChange.resourceType,
                    scrutinyType: cfnspec.propertySpecification(resourceChange.resourceType, propertyName).ScrutinyType,
                    oldValue: resourceChange.oldProperties && resourceChange.oldProperties[propertyName],
                    newValue: resourceChange.newProperties && resourceChange.newProperties[propertyName],
                });
            }
        }
        return ret;
    }
    /**
     * Return all resource changes of a given scrutiny type
     *
     * We don't just look at resource updates; we also look at resource additions and deletions (in which
     * case there is no further detail on property values), and resource type changes.
     */
    scrutinizableResourceChanges(scrutinyTypes) {
        const ret = new Array();
        const scrutinizableTypes = new Set(cfnspec.scrutinizableResourceTypes(scrutinyTypes));
        for (const [resourceLogicalId, resourceChange] of Object.entries(this.resources.changes)) {
            if (!resourceChange) {
                continue;
            }
            const commonProps = {
                oldProperties: resourceChange.oldProperties,
                newProperties: resourceChange.newProperties,
                resourceLogicalId,
            };
            // Even though it's not physically possible in CFN, let's pretend to handle a change of 'Type'.
            if (resourceChange.resourceTypeChanged) {
                // Treat as DELETE+ADD
                if (scrutinizableTypes.has(resourceChange.oldResourceType)) {
                    ret.push({
                        ...commonProps,
                        newProperties: undefined,
                        resourceType: resourceChange.oldResourceType,
                        scrutinyType: cfnspec.resourceSpecification(resourceChange.oldResourceType).ScrutinyType,
                    });
                }
                if (scrutinizableTypes.has(resourceChange.newResourceType)) {
                    ret.push({
                        ...commonProps,
                        oldProperties: undefined,
                        resourceType: resourceChange.newResourceType,
                        scrutinyType: cfnspec.resourceSpecification(resourceChange.newResourceType).ScrutinyType,
                    });
                }
            }
            else {
                if (scrutinizableTypes.has(resourceChange.resourceType)) {
                    ret.push({
                        ...commonProps,
                        resourceType: resourceChange.resourceType,
                        scrutinyType: cfnspec.resourceSpecification(resourceChange.resourceType).ScrutinyType,
                    });
                }
            }
        }
        return ret;
    }
}
exports.TemplateDiff = TemplateDiff;
/**
 * Models an entity that changed between two versions of a CloudFormation template.
 */
class Difference {
    /**
     * @param oldValue the old value, cannot be equal (to the sense of +deepEqual+) to +newValue+.
     * @param newValue the new value, cannot be equal (to the sense of +deepEqual+) to +oldValue+.
     */
    constructor(oldValue, newValue) {
        this.oldValue = oldValue;
        this.newValue = newValue;
        if (oldValue === undefined && newValue === undefined) {
            throw new assert_1.AssertionError({ message: 'oldValue and newValue are both undefined!' });
        }
        this.isDifferent = !util_1.deepEqual(oldValue, newValue);
    }
    /** @returns +true+ if the element is new to the template. */
    get isAddition() {
        return this.oldValue === undefined;
    }
    /** @returns +true+ if the element was removed from the template. */
    get isRemoval() {
        return this.newValue === undefined;
    }
    /** @returns +true+ if the element was already in the template and is updated. */
    get isUpdate() {
        return this.oldValue !== undefined
            && this.newValue !== undefined;
    }
}
exports.Difference = Difference;
class PropertyDifference extends Difference {
    constructor(oldValue, newValue, args) {
        super(oldValue, newValue);
        this.changeImpact = args.changeImpact;
    }
}
exports.PropertyDifference = PropertyDifference;
class DifferenceCollection {
    constructor(diffs) {
        this.diffs = diffs;
    }
    get changes() {
        return onlyChanges(this.diffs);
    }
    get differenceCount() {
        return Object.values(this.changes).length;
    }
    get(logicalId) {
        const ret = this.diffs[logicalId];
        if (!ret) {
            throw new Error(`No object with logical ID '${logicalId}'`);
        }
        return ret;
    }
    get logicalIds() {
        return Object.keys(this.changes);
    }
    /**
     * Returns a new TemplateDiff which only contains changes for which `predicate`
     * returns `true`.
     */
    filter(predicate) {
        const newChanges = {};
        for (const id of Object.keys(this.changes)) {
            const diff = this.changes[id];
            if (predicate(diff)) {
                newChanges[id] = diff;
            }
        }
        return new DifferenceCollection(newChanges);
    }
    /**
     * Invokes `cb` for all changes in this collection.
     *
     * Changes will be sorted as follows:
     *  - Removed
     *  - Added
     *  - Updated
     *  - Others
     *
     * @param cb
     */
    forEachDifference(cb) {
        const removed = new Array();
        const added = new Array();
        const updated = new Array();
        const others = new Array();
        for (const logicalId of this.logicalIds) {
            const change = this.changes[logicalId];
            if (change.isAddition) {
                added.push({ logicalId, change });
            }
            else if (change.isRemoval) {
                removed.push({ logicalId, change });
            }
            else if (change.isUpdate) {
                updated.push({ logicalId, change });
            }
            else if (change.isDifferent) {
                others.push({ logicalId, change });
            }
        }
        removed.forEach(v => cb(v.logicalId, v.change));
        added.forEach(v => cb(v.logicalId, v.change));
        updated.forEach(v => cb(v.logicalId, v.change));
        others.forEach(v => cb(v.logicalId, v.change));
    }
}
exports.DifferenceCollection = DifferenceCollection;
class ConditionDifference extends Difference {
}
exports.ConditionDifference = ConditionDifference;
class MappingDifference extends Difference {
}
exports.MappingDifference = MappingDifference;
class MetadataDifference extends Difference {
}
exports.MetadataDifference = MetadataDifference;
class OutputDifference extends Difference {
}
exports.OutputDifference = OutputDifference;
class ParameterDifference extends Difference {
}
exports.ParameterDifference = ParameterDifference;
var ResourceImpact;
(function (ResourceImpact) {
    /** The existing physical resource will be updated */
    ResourceImpact["WILL_UPDATE"] = "WILL_UPDATE";
    /** A new physical resource will be created */
    ResourceImpact["WILL_CREATE"] = "WILL_CREATE";
    /** The existing physical resource will be replaced */
    ResourceImpact["WILL_REPLACE"] = "WILL_REPLACE";
    /** The existing physical resource may be replaced */
    ResourceImpact["MAY_REPLACE"] = "MAY_REPLACE";
    /** The existing physical resource will be destroyed */
    ResourceImpact["WILL_DESTROY"] = "WILL_DESTROY";
    /** The existing physical resource will be removed from CloudFormation supervision */
    ResourceImpact["WILL_ORPHAN"] = "WILL_ORPHAN";
    /** There is no change in this resource */
    ResourceImpact["NO_CHANGE"] = "NO_CHANGE";
})(ResourceImpact = exports.ResourceImpact || (exports.ResourceImpact = {}));
/**
 * This function can be used as a reducer to obtain the resource-level impact of a list
 * of property-level impacts.
 * @param one the current worst impact so far.
 * @param two the new impact being considered (can be undefined, as we may not always be
 *      able to determine some peroperty's impact).
 */
function worstImpact(one, two) {
    if (!two) {
        return one;
    }
    const badness = {
        [ResourceImpact.NO_CHANGE]: 0,
        [ResourceImpact.WILL_UPDATE]: 1,
        [ResourceImpact.WILL_CREATE]: 2,
        [ResourceImpact.WILL_ORPHAN]: 3,
        [ResourceImpact.MAY_REPLACE]: 4,
        [ResourceImpact.WILL_REPLACE]: 5,
        [ResourceImpact.WILL_DESTROY]: 6,
    };
    return badness[one] > badness[two] ? one : two;
}
/**
 * Change to a single resource between two CloudFormation templates
 *
 * This class can be mutated after construction.
 */
class ResourceDifference {
    constructor(oldValue, newValue, args) {
        this.oldValue = oldValue;
        this.newValue = newValue;
        this.resourceTypes = args.resourceType;
        this.propertyDiffs = args.propertyDiffs;
        this.otherDiffs = args.otherDiffs;
        this.isAddition = oldValue === undefined;
        this.isRemoval = newValue === undefined;
    }
    get oldProperties() {
        return this.oldValue && this.oldValue.Properties;
    }
    get newProperties() {
        return this.newValue && this.newValue.Properties;
    }
    /**
     * Whether this resource was modified at all
     */
    get isDifferent() {
        return this.differenceCount > 0 || this.oldResourceType !== this.newResourceType;
    }
    /**
     * Whether the resource was updated in-place
     */
    get isUpdate() {
        return this.isDifferent && !this.isAddition && !this.isRemoval;
    }
    get oldResourceType() {
        return this.resourceTypes.oldType;
    }
    get newResourceType() {
        return this.resourceTypes.newType;
    }
    /**
     * All actual property updates
     */
    get propertyUpdates() {
        return onlyChanges(this.propertyDiffs);
    }
    /**
     * All actual "other" updates
     */
    get otherChanges() {
        return onlyChanges(this.otherDiffs);
    }
    /**
     * Return whether the resource type was changed in this diff
     *
     * This is not a valid operation in CloudFormation but to be defensive we're going
     * to be aware of it anyway.
     */
    get resourceTypeChanged() {
        return (this.resourceTypes.oldType !== undefined
            && this.resourceTypes.newType !== undefined
            && this.resourceTypes.oldType !== this.resourceTypes.newType);
    }
    /**
     * Return the resource type if it was unchanged
     *
     * If the resource type was changed, it's an error to call this.
     */
    get resourceType() {
        if (this.resourceTypeChanged) {
            throw new Error('Cannot get .resourceType, because the type was changed');
        }
        return this.resourceTypes.oldType || this.resourceTypes.newType;
    }
    /**
     * Replace a PropertyChange in this object
     *
     * This affects the property diff as it is summarized to users, but it DOES
     * NOT affect either the "oldValue" or "newValue" values; those still contain
     * the actual template values as provided by the user (they might still be
     * used for downstream processing).
     */
    setPropertyChange(propertyName, change) {
        this.propertyDiffs[propertyName] = change;
    }
    get changeImpact() {
        // Check the Type first
        if (this.resourceTypes.oldType !== this.resourceTypes.newType) {
            if (this.resourceTypes.oldType === undefined) {
                return ResourceImpact.WILL_CREATE;
            }
            if (this.resourceTypes.newType === undefined) {
                return this.oldValue.DeletionPolicy === 'Retain'
                    ? ResourceImpact.WILL_ORPHAN
                    : ResourceImpact.WILL_DESTROY;
            }
            return ResourceImpact.WILL_REPLACE;
        }
        // Base impact (before we mix in the worst of the property impacts);
        // WILL_UPDATE if we have "other" changes, NO_CHANGE if there are no "other" changes.
        const baseImpact = Object.keys(this.otherChanges).length > 0 ? ResourceImpact.WILL_UPDATE : ResourceImpact.NO_CHANGE;
        return Object.values(this.propertyDiffs)
            .map(elt => elt.changeImpact)
            .reduce(worstImpact, baseImpact);
    }
    /**
     * Count of actual differences (not of elements)
     */
    get differenceCount() {
        return Object.values(this.propertyUpdates).length
            + Object.values(this.otherChanges).length;
    }
    /**
     * Invoke a callback for each actual difference
     */
    forEachDifference(cb) {
        for (const key of Object.keys(this.propertyUpdates).sort()) {
            cb('Property', key, this.propertyUpdates[key]);
        }
        for (const key of Object.keys(this.otherChanges).sort()) {
            cb('Other', key, this.otherDiffs[key]);
        }
    }
}
exports.ResourceDifference = ResourceDifference;
function isPropertyDifference(diff) {
    return diff.changeImpact !== undefined;
}
exports.isPropertyDifference = isPropertyDifference;
/**
 * Filter a map of IDifferences down to only retain the actual changes
 */
function onlyChanges(xs) {
    const ret = {};
    for (const [key, diff] of Object.entries(xs)) {
        if (diff.isDifferent) {
            ret[key] = diff;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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