from json import dumps as json_dumps
from json import loads as json_loads
from math import log10
from os import get_terminal_size
from time import sleep
from typing import Any
from typing import Callable
from typing import Dict
from typing import List
from typing import Optional
from typing import Tuple
from typing import Union

from faapi import DisabledAccount
from faapi import FAAPI
from faapi import Journal
from faapi import NoticeMessage
from faapi import ParsingError
from faapi import Submission
from faapi import SubmissionPartial
from falocalrepo_database import FADatabase

from .commands import Bar
from .commands import clean_string
from .commands import clean_username


class UnknownFolder(Exception):
    pass


def read_cookies(db: FADatabase) -> List[Dict[str, str]]:
    return [{"name": n, "value": v} for n, v in json_loads(db.settings["COOKIES"]).items()]


def write_cookies(db: FADatabase, **cookies):
    db.settings["COOKIES"] = json_dumps(cookies)
    db.commit()


def load_api(db: FADatabase) -> FAAPI:
    print((s := "Connecting... "), end="", flush=True)
    api: FAAPI = FAAPI(read_cookies(db))
    print("\r" + (" " * len(s)), end="\r", flush=True)

    if not api.connection_status:
        raise ConnectionError("FAAPI cannot connect to FA")

    return api


def download_items(db: FADatabase, item_ids: List[str], f: Callable[[FAAPI, FADatabase, int], Any]):
    if item_ids_fail := list(filter(lambda i: not i.isdigit(), item_ids)):
        print("The following ID's are not correct:", *item_ids_fail)
    item_ids = list(filter(lambda i: i.isdigit(), item_ids))
    api: Optional[FAAPI] = load_api(db) if item_ids else None
    for item_id in map(int, filter(lambda i: i.isdigit(), item_ids)):
        print(f"Downloading {item_id:010} ", end="", flush=True)
        f(api, db, item_id)


def save_submission(db: FADatabase, sub: Submission, sub_file: Optional[bytes], user_update: bool = False):
    sub_dict: dict = dict(sub)
    sub_dict["filelink"] = sub_dict["file_url"]
    del sub_dict["file_url"]
    db.submissions.save_submission({**sub_dict, "USERUPDATE": int(user_update)}, sub_file)
    db.commit()


def download_submission_file(api: FAAPI, sub_file_url: str, speed: int = 100) -> Optional[bytes]:
    bar: Bar = Bar(10)
    file_binary: Optional[bytes] = bytes()

    try:
        with api.session.get(sub_file_url, stream=True) as file_stream:
            file_stream.raise_for_status()
            size: int = int(file_stream.headers.get("Content-Length", 0))
            if not size:
                file_binary = file_stream.content
            else:
                for chunk in file_stream.iter_content(chunk_size=1024):
                    file_binary += chunk
                    bar.update(size, len(file_binary)) if size else None
                    sleep(1 / speed) if speed > 0 else None

            bar.update(1, 1)
    except KeyboardInterrupt:
        print("\b\b  \b\b", end="")
        bar.delete()
        bar.__init__(bar.length)
        bar.message("INTERRUPT")
        raise
    except (Exception, BaseException):
        bar.message("FILE ERR")
        file_binary = None
    finally:
        bar.close()

    return file_binary


def download_submission(api: FAAPI, db: FADatabase, sub_id: int, user_update: bool = False) -> bool:
    try:
        sub: Submission = api.get_submission(sub_id, False)[0]
        sub_file: Optional[bytes] = download_submission_file(api, sub.file_url)
        save_submission(db, sub, sub_file, user_update)
        return True
    except ParsingError:
        return False


def download_submissions(db: FADatabase, sub_ids: List[str]):
    download_items(db, sub_ids, download_submission)


def save_journal(db: FADatabase, journal: Journal, user_update: bool = False):
    db.journals.save_journal({**dict(journal), "USERUPDATE": int(user_update)})
    db.commit()


def download_journal(api: FAAPI, db: FADatabase, jrn_id: int):
    journal: Journal = api.get_journal(jrn_id)
    save_journal(db, journal)


def download_journals(db: FADatabase, jrn_ids: List[str]):
    download_items(db, jrn_ids, download_journal)


def download_users_update(db: FADatabase, users: List[str], folders: List[str], stop: int = 1):
    api: Optional[FAAPI] = None
    tot, fail = 0, 0

    users = list(set(map(clean_username, users)))
    users_db: List[dict] = sorted(
        filter(lambda u: u["USERNAME"] in users, db.users),
        key=lambda u: users.index(u["USERNAME"]))

    for user, user_folders in ((u["USERNAME"], u["FOLDERS"].split(",")) for u in users_db):
        if any(folder.startswith("!") for folder in user_folders):
            print(f"User {user} disabled")
            continue
        elif not (user_folders := [f for f in folders if f in user_folders] if folders else user_folders):
            continue
        try:
            api = load_api(db) if api is None else api
            for folder in user_folders:
                print(f"Updating: {user}/{folder}")
                tot_, fail_ = download_user(api, db, user, folder, stop)
                tot += tot_
                fail += fail_
        except DisabledAccount:
            print(f"User {user} disabled")
            db.users.disable_user(user)
            db.commit()
        except NoticeMessage:
            print(f"User {user} not found")
        except ParsingError as err:
            print(f"User {user} error: {repr(err)}")
            continue

    print("Items downloaded:", tot)
    print("Items failed:", fail) if fail else None


def download_users(db: FADatabase, users: List[str], folders: List[str]):
    api: Optional[FAAPI] = None
    for user in users:
        user_is_new: bool = user not in db.users
        try:
            api = load_api(db) if api is None else api
            for folder in folders:
                print(f"Downloading: {user}/{folder}")
                tot, fail = download_user(api, db, user, folder)
                print("Items downloaded:", tot)
                print("Items failed:", fail) if fail else None
        except DisabledAccount:
            print(f"User {user} disabled")
            db.users.disable_user(user)
            db.commit()
        except NoticeMessage:
            print(f"User {user} not found")
            if user_is_new:
                del db.users[user]
                db.commit()
        except ParsingError as err:
            print(f"User {user} error: {repr(err)}")


def download_user(api: FAAPI, db: FADatabase, user: str, folder: str, stop: int = 0) -> Tuple[int, int]:
    items_total: int = 0
    items_failed: int = 0
    page: Union[int, str] = 1
    page_n: int = 0
    user = clean_username(user)
    space_bar: int = 10
    space_term: int = get_terminal_size()[0]
    space_line: int = space_term - (space_bar + 2 + 2)
    found_items: int = 0
    skip: bool = False

    download: Callable[[str, Union[str, int]], Tuple[List[Union[SubmissionPartial, Journal]], Union[int, str]]]
    exists: Callable[[int], Optional[dict]]

    if folder.startswith("!"):
        print(f"{user}/{folder} disabled")
        return 0, 0
    elif folder in ("gallery", "list-gallery"):
        download = api.gallery
        exists = db.submissions.__getitem__
    elif folder in ("scraps", "list-scraps"):
        download = api.scraps
        exists = db.submissions.__getitem__
    elif folder in ("favorites", "list-favorites"):
        page = "next"
        download = api.favorites
        exists = db.submissions.__getitem__
    elif folder in ("journals", "list-journals"):
        download = api.journals
        exists = db.journals.__getitem__
    else:
        raise UnknownFolder(folder)

    if folder.startswith("list-"):
        skip = True
        folder = folder[5:]  # remove list- prefix
    else:
        db.users.new_user(user)
        db.users.add_user_folder(user, folder)
        db.commit()

    while page:
        page_n += 1
        print(f"{page_n}    {user[:space_term - int(log10(page_n)) - 8 - 1]} ...", end="", flush=True)
        try:
            items, page = download(user, page)
        except (Exception, BaseException):
            raise
        finally:
            print("\r" + (" " * (space_term - 1)), end="\r", flush=True)
        for i, item in enumerate(items, 1):
            sub_string: str = f"{page_n}/{i:02d} {item.id:010d} {clean_string(item.title)}"
            print(f"{sub_string[:space_line]:<{space_line}} ", end="", flush=True)
            bar: Bar = Bar(space_bar)
            if not item.id:
                items_failed += 1
                bar.message("ID ERROR")
                bar.close()
            elif item_ := exists(item.id):
                bar.message("IS IN DB")
                if folder == "favorites":
                    found_items += db.submissions.add_favorite(item.id, user)
                    db.commit()
                else:
                    found_items += item_["USERUPDATE"]
                    if folder in ("gallery", "scraps"):
                        db.submissions.update({"USERUPDATE": 1}, item.id)
                        db.submissions.set_folder(item.id, folder)
                    elif folder == "journals":
                        db.journals.update({"USERUPDATE": 1}, item.id)
                    db.commit()
                if stop and found_items >= stop:
                    print("\r" + (" " * (space_term - 1)), end="\r", flush=True)
                    page = 0
                    break
                bar.close()
            elif skip:
                bar.message("SKIPPED")
                bar.close()
            elif isinstance(item, SubmissionPartial):
                bar.delete()
                if download_submission(api, db, item.id, folder != "favorites"):
                    if folder == "favorites":
                        db.submissions.add_favorite(item.id, user)
                        db.commit()
                    items_total += 1
            elif isinstance(item, Journal):
                save_journal(db, item, folder == "journals")
                bar.update(1, 1)
                bar.close()
                items_total += 1

    return items_total, items_failed
