"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const rule_1 = require("../lib/rule");
class SomeTarget {
    constructor(input) {
        this.input = input;
    }
    bind() {
        return { id: 'T1', arn: 'ARN1', input: this.input };
    }
}
module.exports = {
    'json template': {
        'can just be a JSON object'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromObject({ SomeObject: 'withAValue' })));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: '{"SomeObject":"withAValue"}',
                    },
                ],
            }));
            test.done();
        },
        'can use joined JSON containing refs in JSON object'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromObject({
                data: lib_1.EventField.fromPath('$'),
                stackName: cdk.Aws.STACK_NAME,
            })));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        InputTransformer: {
                            InputPathsMap: {
                                f1: '$',
                            },
                            InputTemplate: {
                                'Fn::Join': [
                                    '',
                                    [
                                        '{"data":<f1>,"stackName":"',
                                        { Ref: 'AWS::StackName' },
                                        '"}',
                                    ],
                                ],
                            },
                        },
                    },
                ],
            }));
            test.done();
        },
        'can use token'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            const user = new aws_iam_1.User(stack, 'User');
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromObject({ userArn: user.userArn })));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: {
                            'Fn::Join': [
                                '',
                                [
                                    '{\"userArn\":\"',
                                    {
                                        'Fn::GetAtt': [
                                            'User00B015A1',
                                            'Arn',
                                        ],
                                    },
                                    '\"}',
                                ],
                            ],
                        },
                    },
                ],
            }));
            test.done();
        },
    },
    'text templates': {
        'strings with newlines are serialized to a newline-delimited list of JSON strings'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromMultilineText('I have\nmultiple lines')));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: '"I have"\n"multiple lines"',
                    },
                ],
            }));
            test.done();
        },
        'escaped newlines are not interpreted as newlines'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromMultilineText('this is not\\na real newline'))),
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                    Targets: [
                        {
                            Input: '"this is not\\\\na real newline"',
                        },
                    ],
                }));
            test.done();
        },
        'can use Tokens in text templates'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
            });
            const world = cdk.Lazy.stringValue({ produce: () => 'world' });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromText(`hello ${world}`)));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: '"hello world"',
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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