import { IResource, Resource } from 'aws-cdk-lib/core';
import { Construct } from 'constructs';
/**
 * An API Key
 */
export interface IApiKey extends IResource {
    /**
     * The name of the api key
     *
     * @attribute
     */
    readonly apiKeyName: string;
    /**
     * The Amazon Resource Name (ARN) of the api key resource
     *
     * @attribute Arn, apiKeyArn
     */
    readonly apiKeyArn: string;
}
/**
 * Properties for an API Key
 */
export interface ApiKeyProps {
    /**
     * A name for the api key
     *
     * Must be between 1 and 100 characters and contain only alphanumeric characters,
     * hyphens, periods and underscores.
     *
     * Must be a unique API key name.
     *
     * @default - A name is automatically generated
     */
    readonly apiKeyName?: string;
    /**
     * A description for the api key
     *
     * @default - no description
     */
    readonly description?: string;
    /**
     * The optional timestamp for when the API key resource will expire
     *
     * `expireTime` must be set when `noExpiry` is false or undefined.
     * When `expireTime` is not set, `noExpiry` must be `true`.
     *
     * @default undefined - The API Key never expires
     */
    readonly expireTime?: Date;
    /**
     * `forceDelete` bypasses an API key's expiry conditions and deletes the key.
     * Set the parameter true to delete the key or to false to not preemptively delete the API key.
     *
     * @default undefined - not force delete
     */
    readonly forceDelete?: boolean;
    /**
     * The boolean flag to be included for updating ExpireTime or Restrictions details.
     * Must be set to true to update an API key resource that has been used in the past 7 days.
     * False if force update is not preferred.
     *
     * @default undefined - not force update
     */
    readonly forceUpdate?: boolean;
    /**
     * Whether the API key should expire.
     *
     * Set to `true` when `expireTime` is not set.
     * When you set `expireTime`, `noExpiry` must be `false` or `undefined`.
     *
     * @default undefined - The API Key expires at `expireTime`
     */
    readonly noExpiry?: boolean;
    /**
     * A list of allowed actions for Maps that an API key resource grants permissions to perform.
     *
     * @default - no actions for Maps are permitted
     */
    readonly allowMapsActions?: AllowMapsAction[];
    /**
     * A list of allowed actions for Places that an API key resource grants permissions to perform.
     *
     * @default - no actions for Places are permitted
     */
    readonly allowPlacesActions?: AllowPlacesAction[];
    /**
     * A list of allowed actions for Routes that an API key resource grants permissions to perform.
     *
     * @default - no actions for Routes are permitted
     */
    readonly allowRoutesActions?: AllowRoutesAction[];
    /**
     * An optional list of allowed HTTP referers for which requests must originate from.
     * Requests using this API key from other domains will not be allowed.
     *
     * @see https://docs.aws.amazon.com/ja_jp/AWSCloudFormation/latest/UserGuide/aws-properties-location-apikey-apikeyrestrictions.html#cfn-location-apikey-apikeyrestrictions-allowreferers
     * @default - no Referer
     */
    readonly allowReferers?: string[];
}
/**
 * Actions for Maps that an API key resource grants permissions to perform.
 *
 * @see https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlocationservicemaps.html
 */
export declare enum AllowMapsAction {
    /**
     * Allows getting static map images.
     */
    GET_STATIC_MAP = "geo-maps:GetStaticMap",
    /**
     * Allows getting map tiles for rendering.
     */
    GET_TILE = "geo-maps:GetTile",
    /**
     * Allows any maps actions.
     */
    ANY = "geo-maps:*"
}
/**
 * Actions for Places that an API key resource grants permissions to perform.
 *
 * @see https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlocationserviceplaces.html
 */
export declare enum AllowPlacesAction {
    /**
     * Allows auto-completion of search text.
     */
    AUTOCOMPLETE = "geo-places:Autocomplete",
    /**
     * Allows finding geo coordinates of a known place.
     */
    GEOCODE = "geo-places:Geocode",
    /**
     * Allows getting details of a place.
     */
    GET_PLACE = "geo-places:GetPlace",
    /**
     * Allows getting nearest address to geo coordinates.
     */
    REVERSE_GEOCODE = "geo-places:ReverseGeocode",
    /**
     * Allows category based places search around geo coordinates.
     */
    SEARCH_NEARBY = "geo-places:SearchNearby",
    /**
     * Allows place or address search based on free-form text.
     */
    SEARCH_TEXT = "geo-places:SearchText",
    /**
     * Allows suggestions based on an incomplete or misspelled query.
     */
    SUGGEST = "geo-places:Suggest",
    /**
     * Allows any places actions.
     */
    ANY = "geo-places:*"
}
/**
 * Actions for Routes that an API key resource grants permissions to perform.
 *
 * @see https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlocationserviceroutes.html
 */
export declare enum AllowRoutesAction {
    /**
     * Allows isoline calculation.
     */
    CALCULATE_ISOLINES = "geo-routes:CalculateIsolines",
    /**
     * Allows point to point routing.
     */
    CALCULATE_ROUTES = "geo-routes:CalculateRoutes",
    /**
     * Allows matrix routing.
     */
    CALCULATE_ROUTE_MATRIX = "geo-routes:CalculateRouteMatrix",
    /**
     * Allows computing the best sequence of waypoints.
     */
    OPTIMIZE_WAYPOINTS = "geo-routes:OptimizeWaypoints",
    /**
     * Allows snapping GPS points to a likely route.
     */
    SNAP_TO_ROADS = "geo-routes:SnapToRoads",
    /**
     * Allows any routes actions.
     */
    ANY = "geo-routes:*"
}
/**
 * An API Key
 *
 * @see https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html
 */
export declare class ApiKey extends Resource implements IApiKey {
    /** Uniquely identifies this class. */
    static readonly PROPERTY_INJECTION_ID: string;
    /**
     * Use an existing api key by name
     */
    static fromApiKeyName(scope: Construct, id: string, apiKeyName: string): IApiKey;
    /**
     * Use an existing api key by ARN
     */
    static fromApiKeyArn(scope: Construct, id: string, apiKeyArn: string): IApiKey;
    readonly apiKeyName: string;
    readonly apiKeyArn: string;
    /**
     * The timestamp for when the api key resource was created in ISO 8601 format
     *
     * @attribute
     */
    readonly apiKeyCreateTime: string;
    /**
     * The timestamp for when the api key resource was last updated in ISO 8601 format
     *
     * @attribute
     */
    readonly apiKeyUpdateTime: string;
    constructor(scope: Construct, id: string, props?: ApiKeyProps);
    /**
     * Renders the restrictions property for API Keys
     *
     * NOTE: added allowResources are AWS managed resources.
     */
    private _renderRestrictions;
}
