import * as iam from 'aws-cdk-lib/aws-iam';
import { IResource, Resource } from 'aws-cdk-lib/core';
import { Construct } from 'constructs';
import { DataSource } from './util';
/**
 * A Place Index
 */
export interface IPlaceIndex extends IResource {
    /**
     * The name of the place index
     *
     * @attribute
     */
    readonly placeIndexName: string;
    /**
     * The Amazon Resource Name (ARN) of the place index resource
     *
     * @attribute Arn,IndexArn
     */
    readonly placeIndexArn: string;
}
/**
 * Properties for a place index
 */
export interface PlaceIndexProps {
    /**
     * A name for the place index
     *
     * Must be between 1 and 100 characters and contain only alphanumeric characters,
     * hyphens, periods and underscores.
     *
     * @default - A name is automatically generated
     */
    readonly placeIndexName?: string;
    /**
     * Data source for the place index
     *
     * @default DataSource.ESRI
     */
    readonly dataSource?: DataSource;
    /**
     * Intend use for the results of an operation
     *
     * @default IntendedUse.SINGLE_USE
     */
    readonly intendedUse?: IntendedUse;
    /**
     * A description for the place index
     *
     * @default - no description
     */
    readonly description?: string;
}
/**
 * Intend use for the results of an operation
 */
export declare enum IntendedUse {
    /**
     * The results won't be stored
     */
    SINGLE_USE = "SingleUse",
    /**
     * The result can be cached or stored in a database
     */
    STORAGE = "Storage"
}
/**
 * A Place Index
 *
 * @see https://docs.aws.amazon.com/location/latest/developerguide/places-concepts.html
 */
export declare class PlaceIndex extends Resource implements IPlaceIndex {
    /** Uniquely identifies this class. */
    static readonly PROPERTY_INJECTION_ID: string;
    /**
     * Use an existing place index by name
     */
    static fromPlaceIndexName(scope: Construct, id: string, placeIndexName: string): IPlaceIndex;
    /**
     * Use an existing place index by ARN
     */
    static fromPlaceIndexArn(scope: Construct, id: string, placeIndexArn: string): IPlaceIndex;
    readonly placeIndexName: string;
    readonly placeIndexArn: string;
    /**
     * The timestamp for when the place index resource was created in ISO 8601 format
     *
     * @attribute
     */
    readonly placeIndexCreateTime: string;
    /**
     * The timestamp for when the place index resource was last updated in ISO 8601 format
     *
     * @attribute
     */
    readonly placeIndexUpdateTime: string;
    constructor(scope: Construct, id: string, props?: PlaceIndexProps);
    /**
     * Grant the given principal identity permissions to perform the actions on this place index.
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to search using this index
     */
    grantSearch(grantee: iam.IGrantable): iam.Grant;
}
