"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Archiver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * Every week
 */
const DEFAULT_CRON_EXPRESSION = 'cron(0 0 ? * 1 *)';
class Archiver extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.kmsKey = this.createKey();
        this.logGroup = this.createLogGroup();
        this.topic = new aws_cdk_lib_1.aws_sns.Topic(this, 'notifications', {
            displayName: 'archiver-notifications',
            masterKey: this.kmsKey,
        });
        this.bucket = this.createArchiveBucket();
        this.createS3Notifications();
        this.createProjects();
        this.createCfnOutputs();
    }
    /**
     * Set up the S3-related event notifcations.
     *
     * @private
     * @memberof Archiver
     */
    createS3Notifications() {
        if (this.props.notificationEvents) {
            this.props.notificationEvents.forEach((event) => {
                this.bucket.addEventNotification(event, new aws_cdk_lib_1.aws_s3_notifications.SnsDestination(this.topic));
            });
        }
    }
    createCfnOutputs() {
        new aws_cdk_lib_1.CfnOutput(this, 's3-bucket-arn', {
            description: 'ARN of the S3 bucket storing the repositories.',
            value: this.bucket.bucketArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'log-group-arn', {
            description: 'ARN of the Cloudwatch Log group storing the code build logs.',
            value: this.logGroup.logGroupArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'kms-key', {
            description: 'ARN of the KMS key used to encrypt the Cloudwatch logs and the SNS topic.',
            value: this.kmsKey.keyArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'sns-topic-arn', {
            description: 'ARN of the SNS topic.',
            value: this.topic.topicArn,
        });
    }
    createLogGroup() {
        const loggroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'loggroup', {
            encryptionKey: this.kmsKey,
            retention: this.props.retentionDays
                ? this.props.retentionDays
                : aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        loggroup.node.addDependency(this.kmsKey);
        return loggroup;
    }
    /**
     *Create the S3 bucket that will later store the repositories.
     *
     * @private
     * @return {*}
     * @memberof Archiver
     */
    createArchiveBucket() {
        return new aws_cdk_lib_1.aws_s3.Bucket(this, 'destination', {
            blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
            encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(360),
                    transitions: [
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.GLACIER,
                            transitionAfter: aws_cdk_lib_1.Duration.days(90),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: aws_cdk_lib_1.Duration.days(180),
                        },
                    ],
                },
            ],
            publicReadAccess: false,
            versioned: true,
        });
    }
    createKey() {
        const key = new aws_cdk_lib_1.aws_kms.Key(this, 'loggroupKey', {
            description: 'Azure DevOps git repository archiver',
            enableKeyRotation: true,
            pendingWindow: aws_cdk_lib_1.Duration.days(7),
            keyUsage: aws_cdk_lib_1.aws_kms.KeyUsage.ENCRYPT_DECRYPT,
            keySpec: aws_cdk_lib_1.aws_kms.KeySpec.SYMMETRIC_DEFAULT,
            alias: 'archiver-key',
        });
        key.grantEncryptDecrypt(new aws_cdk_lib_1.aws_iam.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`));
        if (this.props.notificationEvents) {
            key.grantEncryptDecrypt(new aws_cdk_lib_1.aws_iam.ServicePrincipal('s3.amazonaws.com'));
        }
        return key;
    }
    /**
     * Creates for each backup configuration a separate CodeBuild project.
     *
     * @private
     * @memberof Archiver
     */
    createProjects() {
        this.props.backupConfigurations.forEach((configuration) => {
            const project = this.createProject(configuration);
            project.enableBatchBuilds();
            this.bucket.grantReadWrite(project);
            this.createCloudwatchRule(project, configuration);
        });
    }
    createCloudwatchRule(project, element) {
        new aws_cdk_lib_1.aws_events.Rule(this, 'ScheduleRule-' + element.organizationName + '-' + element.projectName, {
            enabled: true,
            schedule: aws_cdk_lib_1.aws_events.Schedule.expression(DEFAULT_CRON_EXPRESSION),
            targets: [new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project)],
            description: 'Trigger for backing up Azure DevOps git repositories of organization ' +
                element.organizationName +
                ' and project ' +
                element.projectName +
                '.',
        });
    }
    /**
     * Create a CodeBuild project.
     *
     * @private
     * @param {BackupConfiguration} element
     * @return {*}
     * @memberof Archiver
     */
    createProject(element) {
        return new aws_cdk_lib_1.aws_codebuild.Project(this, 'archiver-' + element.organizationName + '-' + element.projectName, {
            timeout: aws_cdk_lib_1.Duration.hours(5),
            projectName: 'AzureDevOpsGitBackup' +
                '-' +
                element.organizationName +
                '-' +
                element.projectName,
            description: 'Backup Azure DevOps git repositories to an S3 bucket.',
            checkSecretsInPlainTextEnvVariables: true,
            concurrentBuildLimit: 90,
            environment: {
                environmentVariables: {
                    TOKEN: {
                        value: element.secretArn + ':pat',
                        type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    },
                    ORGANIZATION: { value: element.organizationName },
                    PROJECT: { value: element.projectName },
                },
                buildImage: aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_6_0,
            },
            logging: {
                cloudWatch: {
                    enabled: true,
                    logGroup: this.logGroup,
                },
            },
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject({
                version: 0.2,
                batch: {
                    'fast-fail': false,
                    'build-list': this.createBatchConfiguration(element.repositoryNames),
                },
                phases: {
                    install: {
                        'commands': [
                            'apt install -y git',
                        ],
                        'on-failure': 'ABORT',
                    },
                    pre_build: {
                        'commands': [
                            'if [ -z "${ORGANIZATION}" ]; then echo "Missing environment variable ORGANIZATION."; exit 1 ; fi',
                            'if [ -z "${REPOSITORY}" ]; then echo "Missing environment variable REPOSITORY."; exit 1 ; fi',
                            'if [ -z "${PROJECT}" ]; then echo "Missing environment variable PROJECT."; exit 1 ; fi',
                            'if [ -z "${TOKEN}" ]; then echo "Missing environment variable TOKEN."; exit 1 ; fi',
                        ],
                        'on-failure': 'ABORT',
                    },
                    build: {
                        commands: [
                            'git clone --mirror "https://${TOKEN}@dev.azure.com/${ORGANIZATION}/${PROJECT}/_git/${REPOSITORY}"',
                            'tar czf ${REPOSITORY}.tgz ./${REPOSITORY}.git',
                            'aws s3 cp ./${REPOSITORY}.tgz ' +
                                this.bucket.s3UrlForObject() +
                                '/${ORGANIZATION}/${PROJECT}/${REPOSITORY}.tgz',
                        ],
                    },
                },
            }),
        });
    }
    createBatchConfiguration(repositoryNames) {
        const output = [];
        repositoryNames.forEach((element) => {
            output.push({
                identifier: 'build_' + element.replace(/-/g, '_'),
                env: {
                    variables: {
                        REPOSITORY: element,
                    },
                },
            });
        });
        return output;
    }
}
exports.Archiver = Archiver;
_a = JSII_RTTI_SYMBOL_1;
Archiver[_a] = { fqn: "azure-devops-repository-archiver.Archiver", version: "0.0.23" };
//# sourceMappingURL=data:application/json;base64,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