import typing

from django.conf import settings
from django.db import models
from django.db.models import CheckConstraint, Q
from django.utils.translation import gettext_lazy as _
from phonenumber_field.modelfields import PhoneNumberField

from ob_dj_otp.core.otp.managers import OneTruePairingManager
from ob_dj_otp.utils.model import DjangoModelCleanMixin


class OneTruePairing(DjangoModelCleanMixin, models.Model):
    """OTP represent attempts for login or registration using One True Pairing"""

    class Statuses(models.TextChoices):
        init = "init", _("init")
        used = "used", _("used")

    class Usages(models.TextChoices):
        auth = "auth"
        register = "register"

    phone_number = PhoneNumberField(null=True, blank=True)
    email = models.EmailField(null=True, blank=True)
    status = models.CharField(
        max_length=10, choices=Statuses.choices, default=Statuses.init
    )
    usage = models.CharField(max_length=10, choices=Usages.choices)
    user = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        help_text=_("User field will be populated for OTP Auth"),
    )
    verification_code = models.CharField(
        max_length=10,
        help_text=_(
            "Verification code generated by backend for authentication or registration"
        ),
    )
    meta = models.JSONField(default=dict, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    objects = OneTruePairingManager()

    class Meta:
        constraints = [
            CheckConstraint(
                check=~Q(
                    Q(usage="auth", user__isnull=True)
                    | Q(usage="register", user__isnull=False, status="init")
                ),
                name="check_usage_and_user_compatibility",
            ),
        ]

    def __str__(self):
        return f"{self.phone_number.__str__()}"

    def mark_used(self) -> typing.NoReturn:
        self.status = self.Statuses.used
        self.save()
